<?php

namespace Uncanny_Automator_Pro;

use Uncanny_Automator\Recipe\Action;
use DateTime;

/**
 * Class WP_SETPOSTDATE
 *
 * @package Uncanny_Automator_Pro
 */
class WP_SETPOSTDATE extends Action {

	/**
	 * Setup action method.
	 *
	 * @return void
	 */
	protected function setup_action() {
		$this->set_integration( 'WP' );
		$this->set_action_code( 'SETPOSTDATE' );
		$this->set_action_meta( 'WPPOST' );
		$this->set_requires_user( false );
		$this->set_is_pro( true );
		$this->set_sentence(
			sprintf(
				/* translators: %1$s - Post name */
				esc_html_x( 'Update the published date of {{a post:%1$s}}', 'WordPress', 'uncanny-automator-pro' ),
				$this->get_action_meta()
			)
		);
		$this->set_readable_sentence( esc_html_x( 'Update the published date of {{a post}}', 'WordPress', 'uncanny-automator-pro' ) );
	}

	/**
	 * Define the action's options.
	 *
	 * @return array The options configuration.
	 */
	public function options() {
		$post_options = Automator()->helpers->recipe->wp->options->all_posts(
			esc_html_x( 'Post', 'WordPress', 'uncanny-automator-pro' ),
			$this->get_action_meta(),
			array(
				'token'                   => false,
				'is_ajax'                 => false,
				'any_option'              => false,
				'include_relevant_tokens' => false,
			)
		);

		$post_type_options = array();
		if ( isset( $post_options['options'] ) && is_array( $post_options['options'] ) ) {
			foreach ( $post_options['options'] as $value => $text ) {
				if ( '-1' === $value || -1 === $value ) {
					continue;
				}
				$post_type_options[] = array(
					'value' => $value,
					'text'  => $text,
				);
			}
		}

		return array(
			array(
				'option_code'           => $this->get_action_meta(),
				'label'                 => esc_html_x( 'Post', 'WordPress', 'uncanny-automator-pro' ),
				'input_type'            => 'select',
				'required'              => true,
				'supports_custom_value' => false,
				'options'               => $post_type_options,
			),
			array(
				'option_code'           => 'WPPOSTDATE',
				'label'                 => esc_html_x( 'Published date', 'WordPress', 'uncanny-automator-pro' ),
				'input_type'            => 'date',
				'supports_custom_value' => true,
				'required'              => true,
				'description'           => esc_html_x( 'Select the new published date for this post', 'WordPress', 'uncanny-automator-pro' ),
			),
			array(
				'option_code'           => 'WPPOSTTIME',
				'label'                 => esc_html_x( 'Published time', 'WordPress', 'uncanny-automator-pro' ),
				'input_type'            => 'time',
				'supports_custom_value' => true,
				'required'              => false,
				'description'           => esc_html_x( 'Select the new published time for this post (optional)', 'WordPress', 'uncanny-automator-pro' ),
			),
		);
	}

	/**
	 * Define tokens for this action.
	 *
	 * @return array
	 */
	public function define_tokens() {
		return array(
			'POST_TITLE'                  => array(
				'name' => esc_html_x( 'Post title', 'WordPress', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'POST_UPDATED_DATE_FORMATTED' => array(
				'name' => esc_html_x( 'Updated published date (formatted)', 'WordPress', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
		);
	}

	/**
	 * Process the action.
	 *
	 * @param int $user_id
	 * @param array $action_data
	 * @param int $recipe_id
	 * @param array $args
	 * @param array $parsed
	 */
	protected function process_action( $user_id, $action_data, $recipe_id, $args, $parsed ) {
		$post_id   = absint( $this->get_parsed_meta_value( $this->get_action_meta(), 0 ) );
		$post_date = sanitize_text_field( $this->get_parsed_meta_value( 'WPPOSTDATE', '' ) );
		$post_time = sanitize_text_field( $this->get_parsed_meta_value( 'WPPOSTTIME', '00:00' ) );

		// Validate post exists
		$post = get_post( $post_id );
		if ( ! $post instanceof \WP_Post ) {
			$this->add_log_error(
				sprintf(
					/* translators: %d - Post ID */
					esc_html_x( 'Post with ID %d not found.', 'WordPress', 'uncanny-automator-pro' ),
					$post_id
				)
			);
			return false;
		}

		// Parse and validate date - treat input as local site time
		$date_time_string = $post_date . ' ' . $post_time;

		// Create DateTime object in local timezone
		$timezone = wp_timezone();

		// Try multiple date formats
		$datetime = DateTime::createFromFormat( 'Y-m-d H:i:s', $date_time_string, $timezone );

		if ( false === $datetime ) {
			// Try alternative format with different separators
			$datetime = DateTime::createFromFormat( 'Y-m-d H:i', $date_time_string, $timezone );
		}

		if ( false === $datetime ) {
			// Try with strtotime as fallback
			$timestamp = strtotime( $date_time_string );
			if ( false !== $timestamp ) {
				$datetime = new DateTime( '@' . $timestamp, $timezone );
			}
		}

		if ( false === $datetime ) {
			$this->add_log_error(
				sprintf(
					/* translators: %s - date or time string */
					esc_html_x( 'Invalid date or time format: %s', 'WordPress', 'uncanny-automator-pro' ),
					$date_time_string
				)
			);
			return false;
		}

		// Get timestamp from local datetime
		$timestamp = $datetime->getTimestamp();

		// Format for database - post_date should be local time as entered
		$local_date = $datetime->format( 'Y-m-d H:i:s' );
		// Convert to GMT for post_date_gmt
		$gmt_date = gmdate( 'Y-m-d H:i:s', $timestamp );

		// Prepare update data
		$update_data = array(
			'ID' => $post_id,
		);

		// Update post date with correct timezone handling
		$update_data['post_date']     = $local_date;
		$update_data['post_date_gmt'] = $gmt_date;

		// Update the post
		$result = wp_update_post( $update_data, true );

		if ( is_wp_error( $result ) ) {
			$this->add_log_error( esc_html_x( 'Failed to update post date.', 'WordPress', 'uncanny-automator-pro' ) );
			return false;
		}

		// Hydrate tokens
		$this->hydrate_tokens(
			array(
				'POST_TITLE'                  => get_the_title( $post_id ),
				'POST_UPDATED_DATE_FORMATTED' => get_the_date( '', $post_id ),
			)
		);

		return true;
	}
}
