<?php

namespace Uncanny_Automator_Pro;

use WC_Coupon;
use Uncanny_Automator\Recipe\Trigger;

/**
 * Class Wc_Type_Of_Coupon_Created
 *
 * @package Uncanny_Automator_Pro
 */
class Wc_Type_Of_Coupon_Created extends Trigger {

	const TRIGGER_CODE = 'WC_TYPE_OF_COUPON_CREATED';
	const TRIGGER_META = 'WC_TYPE_OF_COUPON';

	/**
	 * Setup trigger
	 */
	protected function setup_trigger() {
		$this->set_integration( 'WC' );
		$this->set_trigger_code( self::TRIGGER_CODE );
		$this->set_trigger_meta( self::TRIGGER_META );
		$this->set_is_pro( true );
		$this->set_is_login_required( false );
		$this->set_trigger_type( 'anonymous' );
		$this->set_uses_api( false );
		$this->set_sentence(
			// translators: %1$s: type of coupon
			sprintf(
				esc_html_x( 'A {{type of:%1$s}} coupon is created', 'WooCommerce', 'uncanny-automator-pro' ),
				self::TRIGGER_META
			)
		);
		$this->set_readable_sentence( esc_html_x( 'A {{type of}} coupon is created', 'WooCommerce', 'uncanny-automator-pro' ) );
		$this->add_action( 'save_post_shop_coupon', 10, 3 );
	}

	/**
	 * Options
	 */
	public function options() {
		return array(
			array(
				'option_code' => self::TRIGGER_META,
				'label' => esc_html_x( 'Type of coupon', 'WooCommerce', 'uncanny-automator-pro' ),
				'token_name' => esc_html_x( 'Selected coupon type', 'WooCommerce', 'uncanny-automator-pro' ),
				'input_type' => 'select',
				'options' => $this->get_coupon_types(),
				'required' => true,
			),
		);
	}

	/**
	 * Define tokens
	 */
	public function define_tokens( $trigger, $tokens ) {
		return array(
			array(
				'tokenId' => 'WC_COUPON_ID',
				'tokenName' => esc_html_x( 'Coupon ID', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId' => 'WC_COUPON_CODE',
				'tokenName' => esc_html_x( 'Coupon code', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_DESCRIPTION',
				'tokenName' => esc_html_x( 'Coupon description', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_AMOUNT',
				'tokenName' => esc_html_x( 'Coupon amount', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_TYPE',
				'tokenName' => esc_html_x( 'Coupon type', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_USAGE_LIMIT',
				'tokenName' => esc_html_x( 'Usage limit', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId' => 'WC_COUPON_USAGE_LIMIT_PER_USER',
				'tokenName' => esc_html_x( 'Usage limit per user', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId' => 'WC_COUPON_EXPIRY_DATE',
				'tokenName' => esc_html_x( 'Expiry date', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_MINIMUM_AMOUNT',
				'tokenName' => esc_html_x( 'Minimum amount', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_MAXIMUM_AMOUNT',
				'tokenName' => esc_html_x( 'Maximum amount', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_FREE_SHIPPING',
				'tokenName' => esc_html_x( 'Free shipping', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_PRODUCT_IDS',
				'tokenName' => esc_html_x( 'Product IDs', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_EXCLUDED_PRODUCT_IDS',
				'tokenName' => esc_html_x( 'Excluded product IDs', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_PRODUCT_CATEGORIES',
				'tokenName' => esc_html_x( 'Product categories', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_EXCLUDED_PRODUCT_CATEGORIES',
				'tokenName' => esc_html_x( 'Excluded product categories', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId' => 'WC_COUPON_EMAIL_RESTRICTIONS',
				'tokenName' => esc_html_x( 'Email restrictions', 'WooCommerce', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Validate trigger
	 */
	public function validate( $trigger, $hook_args ) {
		list( $coupon_id, $post, $update ) = $hook_args;
		$coupon                            = new WC_Coupon( $coupon_id );

		if ( ! $coupon instanceof WC_Coupon ) {
			return false;
		}

		// Check if coupon is published for first time
		if ( 'publish' !== $coupon->get_status() ) {
			return false;
		}

		$date_created  = $coupon->get_date_created();
		$date_modified = $coupon->get_date_modified();

		if ( $date_created->date( 'Y-m-d H:i:s' ) !== $date_modified->date( 'Y-m-d H:i:s' ) ) {
			return false;
		}

		$selected_type = $trigger['meta'][ self::TRIGGER_META ];
		$coupon_type   = $coupon->get_discount_type();

		// Check if coupon type matches selected type
		return ( '-1' === $selected_type || $coupon_type === $selected_type );
	}

	/**
	 * Hydrate tokens
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		list( $coupon_id, $post, $is_updated ) = $hook_args;

		$coupon = new WC_Coupon( $coupon_id );
		if ( ! $coupon instanceof WC_Coupon ) {
			return array();
		}

		$tokens = array();

		// Basic coupon information
		$tokens['WC_COUPON_ID']                   = $coupon_id;
		$tokens['WC_COUPON_CODE']                 = $coupon->get_code();
		$tokens['WC_COUPON_DESCRIPTION']          = $coupon->get_description();
		$tokens['WC_COUPON_AMOUNT']               = $coupon->get_amount();
		$tokens['WC_COUPON_TYPE']                 = $coupon->get_discount_type();
		$tokens['WC_COUPON_USAGE_LIMIT']          = $coupon->get_usage_limit();
		$tokens['WC_COUPON_USAGE_LIMIT_PER_USER'] = $coupon->get_usage_limit_per_user();

		// Expiry date
		$expiry_date                     = $coupon->get_date_expires();
		$tokens['WC_COUPON_EXPIRY_DATE'] = $expiry_date ? $expiry_date->date( 'Y-m-d H:i:s' ) : '';

		// Amount restrictions
		$tokens['WC_COUPON_MINIMUM_AMOUNT'] = $coupon->get_minimum_amount();
		$tokens['WC_COUPON_MAXIMUM_AMOUNT'] = $coupon->get_maximum_amount();

		// Free shipping
		$tokens['WC_COUPON_FREE_SHIPPING'] = $coupon->get_free_shipping() ? esc_html_x( 'Yes', 'WooCommerce', 'uncanny-automator-pro' ) : esc_html_x( 'No', 'WooCommerce', 'uncanny-automator-pro' );

		// Product restrictions
		$product_ids                     = $coupon->get_product_ids();
		$tokens['WC_COUPON_PRODUCT_IDS'] = ! empty( $product_ids ) ? implode( ', ', $product_ids ) : '';

		$excluded_product_ids                     = $coupon->get_excluded_product_ids();
		$tokens['WC_COUPON_EXCLUDED_PRODUCT_IDS'] = ! empty( $excluded_product_ids ) ? implode( ', ', $excluded_product_ids ) : '';

		// Category restrictions
		$product_categories                     = $coupon->get_product_categories();
		$tokens['WC_COUPON_PRODUCT_CATEGORIES'] = ! empty( $product_categories ) ? implode( ', ', $product_categories ) : '';

		$excluded_product_categories                     = $coupon->get_excluded_product_categories();
		$tokens['WC_COUPON_EXCLUDED_PRODUCT_CATEGORIES'] = ! empty( $excluded_product_categories ) ? implode( ', ', $excluded_product_categories ) : '';

		// Email restrictions
		$email_restrictions                     = $coupon->get_email_restrictions();
		$tokens['WC_COUPON_EMAIL_RESTRICTIONS'] = ! empty( $email_restrictions ) ? implode( ', ', $email_restrictions ) : '';

		// Trigger meta tokens for backward compatibility
		$tokens[ self::TRIGGER_META . '_ID' ] = $coupon_id;
		$tokens[ self::TRIGGER_META ]         = $coupon->get_discount_type();

		return $tokens;
	}

	/**
	 * Get coupon types
	 * Populated dynamically via WooCommerce discount_type values. Use woocommerce_coupon_discount_types filter for full list.
	 */
	private function get_coupon_types() {
		$types = array(
			array(
				'text' => esc_html_x( 'Any type', 'WooCommerce', 'uncanny-automator-pro' ),
				'value' => '-1',
			),
		);

		// Get coupon types from WooCommerce
		$coupon_types = wc_get_coupon_types();

		// Apply filter to get all available discount types
		$discount_types = apply_filters( 'woocommerce_coupon_discount_types', $coupon_types );

		foreach ( $discount_types as $type => $label ) {
			$types[] = array(
				'text' => $label,
				'value' => $type,
			);
		}

		return $types;
	}
}

new Wc_Type_Of_Coupon_Created();
