<?php

namespace Uncanny_Automator_Pro;

/**
 * Class WC_CREATE_SIMPLE_PRODUCT
 *
 * @package Uncanny_Automator_Pro
 */
class WC_CREATE_SIMPLE_PRODUCT extends \Uncanny_Automator\Recipe\Action {

	const ACTION_CODE = 'WC_CREATE_SIMPLE_PRODUCT';
	const ACTION_META = 'WC_SIMPLE_PRODUCT';

	/**
	 * @method \Uncanny_Automator_Pro\Integrations\WooCommerce\WooCommerce_Pro_Helpers get_item_helpers()
	 */

	/**
	 * Setup action
	 */
	protected function setup_action() {
		$this->set_integration( 'WC' );
		$this->set_action_code( self::ACTION_CODE );
		$this->set_action_meta( self::ACTION_META );
		$this->set_is_pro( true );
		$this->set_requires_user( false );
		$this->set_sentence(
			// translators: %1$s: placeholder
			sprintf( esc_html_x( 'Create {{a simple product:%1$s}}', 'WooCommerce', 'uncanny-automator-pro' ), $this->get_action_meta() )
		);
		$this->set_readable_sentence( esc_html_x( 'Create {{a simple product}}', 'WooCommerce', 'uncanny-automator-pro' ) );
	}

	/**
	 * Convert options array to the format expected by the form
	 *
	 * @param array $options
	 * @return array
	 */
	private function convert_options_to_array( $options ) {
		$result = array();
		if ( isset( $options['options'] ) && is_array( $options['options'] ) ) {
			foreach ( $options['options'] as $value => $option ) {
				$result[] = array(
					'value' => $value,
					'text'  => $option,
				);
			}
		}
		return $result;
	}

	/**
	 * Options for the action
	 *
	 * @return array
	 */
	public function options() {
		$product_categories_options = Automator()->helpers->recipe->woocommerce->pro->all_wc_product_categories();
		$product_categories         = $this->convert_options_to_array( $product_categories_options );

		$product_tags_options = Automator()->helpers->recipe->woocommerce->pro->all_wc_product_tags();
		$product_tags         = $this->convert_options_to_array( $product_tags_options );

		return array(
			Automator()->helpers->recipe->field->text(
				array(
					'option_code'           => 'WC_PRODUCT_TITLE',
					'label'                 => esc_html_x( 'Product title', 'WooCommerce', 'uncanny-automator-pro' ),
					'required'              => true,
					'exclude_default_token' => true,
				)
			),
			Automator()->helpers->recipe->field->text(
				array(
					'option_code'           => 'WC_PRODUCT_DESCRIPTION',
					'label'                 => esc_html_x( 'Product description', 'WooCommerce', 'uncanny-automator-pro' ),
					'required'              => false,
					'exclude_default_token' => true,
					'input_type'            => 'textarea',
				)
			),
			Automator()->helpers->recipe->field->text(
				array(
					'option_code'           => 'WC_PRODUCT_SHORT_DESCRIPTION',
					'label'                 => esc_html_x( 'Product short description', 'WooCommerce', 'uncanny-automator-pro' ),
					'required'              => false,
					'exclude_default_token' => true,
					'input_type'            => 'textarea',
				)
			),
			Automator()->helpers->recipe->field->float(
				array(
					'option_code'           => 'WC_PRODUCT_PRICE',
					'label'                 => esc_html_x( 'Price', 'WooCommerce', 'uncanny-automator-pro' ),
					'required'              => true,
					'exclude_default_token' => true,
				)
			),
			Automator()->helpers->recipe->field->float(
				array(
					'option_code'           => 'WC_PRODUCT_SALE_PRICE',
					'label'                 => esc_html_x( 'Sale price', 'WooCommerce', 'uncanny-automator-pro' ),
					'required'              => false,
					'exclude_default_token' => true,
				)
			),
			Automator()->helpers->recipe->field->text(
				array(
					'option_code'           => 'WC_PRODUCT_SKU',
					'label'                 => esc_html_x( 'SKU', 'WooCommerce', 'uncanny-automator-pro' ),
					'required'              => false,
					'exclude_default_token' => true,
				)
			),
			array(
				'input_type'               => 'select',
				'option_code'              => 'WC_PRODUCT_CATEGORIES',
				'label'                    => esc_html_x( 'Product categories', 'WooCommerce', 'uncanny-automator-pro' ),
				'required'                 => false,
				'supports_custom_value'    => false,
				'exclude_default_token'    => true,
				'supports_multiple_values' => true,
				'options'                  => $product_categories,
			),
			array(
				'input_type'               => 'select',
				'option_code'              => 'WC_PRODUCT_TAGS',
				'label'                    => esc_html_x( 'Product tags', 'WooCommerce', 'uncanny-automator-pro' ),
				'supports_custom_value'    => false,
				'required'                 => false,
				'exclude_default_token'    => true,
				'supports_multiple_values' => true,
				'options'                  => $product_tags,
			),
			Automator()->helpers->recipe->field->select(
				array(
					'option_code'           => 'WC_PRODUCT_STATUS',
					'label'                 => esc_html_x( 'Status', 'WooCommerce', 'uncanny-automator-pro' ),
					'exclude_default_token' => true,
					'options'               => array(
						array(
							'value' => 'draft',
							'text'  => 'Draft',
						),
						array(
							'value' => 'publish',
							'text'  => 'Published',
						),
						array(
							'value' => 'pending',
							'text'  => 'Pending Review',
						),
					),
					'default_value'         => 'publish',
					'supports_custom_value' => false,
				)
			),
			Automator()->helpers->recipe->field->text(
				array(
					'option_code'           => 'WC_PRODUCT_IMAGE',
					'exclude_default_token' => true,
					'label'                 => esc_html_x( 'Product image URL', 'WooCommerce', 'uncanny-automator-pro' ),
					'placeholder'           => esc_html_x( 'https://examplewebsite.com/path/to/image.jpg', 'WooCommerce', 'uncanny-automator-pro' ),
					'input_type'            => 'url',
					'required'              => false,
					'description'           => esc_html_x( 'The URL must include a supported image file extension (e.g. .jpg, .png, .svg, etc.). Some sites may block remote image download.', 'WooCommerce', 'uncanny-automator-pro' ),
				)
			),
			array(
				'option_code'           => 'WC_PRODUCT_STOCK_STATUS',
				'input_type'            => 'select',
				'label'                 => esc_html_x( 'Stock status', 'WooCommerce', 'uncanny-automator-pro' ),
				'options'               => Automator()->helpers->recipe->woocommerce->pro->get_all_stock_options(),
				'exclude_default_token' => true,
				'default_value'         => 'instock',
				'supports_custom_value' => false,
			),
			array(
				'option_code'           => 'WC_MANAGE_STOCK',
				'label'                 => esc_html_x( 'Track stock quantity for this product', 'WooCommerce', 'uncanny-automator-pro' ),
				'input_type'            => 'checkbox',
				'is_toggle'             => true,
				'required'              => false,
				'default_value'         => false,
				'exclude_default_token' => true,
			),
			array(
				'option_code'           => 'WC_PRODUCT_STOCK_QUANTITY',
				'input_type'            => 'int',
				'label'                 => esc_html_x( 'Stock quantity', 'WooCommerce', 'uncanny-automator-pro' ),
				'required'              => false,
				'exclude_default_token' => true,
				'description'           => esc_html_x( 'Only used if "Track stock quantity" is enabled', 'WooCommerce', 'uncanny-automator-pro' ),
				'dynamic_visibility'    => array(
					'default_state'    => 'hidden',
					'visibility_rules' => array(
						array(
							'operator'        => 'AND',
							'rule_conditions' => array(
								array(
									'option_code' => 'WC_MANAGE_STOCK',
									'compare'     => '==',
									'value'       => true,
								),
							),
							'resulting_visibility' => 'show',
						),
					),
				),
			),
			array(
				'option_code'           => 'WC_PRODUCT_VIRTUAL',
				'label'                 => esc_html_x( 'Virtual', 'WooCommerce', 'uncanny-automator-pro' ),
				'input_type'            => 'checkbox',
				'is_toggle'             => true,
				'required'              => false,
				'default_value'         => false,
				'exclude_default_token' => true,
				'description'           => esc_html_x( 'Virtual products are intangible and are not shipped.', 'WooCommerce', 'uncanny-automator-pro' ),
			),
			array(
				'option_code'           => 'WC_PRODUCT_DOWNLOADABLE',
				'label'                 => esc_html_x( 'Downloadable', 'WooCommerce', 'uncanny-automator-pro' ),
				'input_type'            => 'checkbox',
				'is_toggle'             => true,
				'required'              => false,
				'default_value'         => false,
				'exclude_default_token' => true,
				'description'           => esc_html_x( 'Downloadable products give access to a file upon purchase.', 'WooCommerce', 'uncanny-automator-pro' ),
			),
			Automator()->helpers->recipe->field->select(
				array(
					'option_code'           => 'WC_PRODUCT_CATALOG_VISIBILITY',
					'label'                 => esc_html_x( 'Catalog visibility', 'WooCommerce', 'uncanny-automator-pro' ),
					'exclude_default_token' => true,
					'options'               => array(
						array(
							'value' => 'visible',
							'text'  => 'Shop and search results',
						),
						array(
							'value' => 'catalog',
							'text'  => 'Shop only',
						),
						array(
							'value' => 'search',
							'text'  => 'Search results only',
						),
						array(
							'value' => 'hidden',
							'text'  => 'Hidden',
						),
					),
					'default_value'         => 'visible',
					'supports_custom_value' => false,
				)
			),
			array(
				'option_code'           => 'IS_FEATURED_PRODUCT',
				'label'                 => esc_html_x( 'This is a featured product', 'WooCommerce', 'uncanny-automator-pro' ),
				'input_type'            => 'checkbox',
				'is_toggle'             => true,
				'required'              => false,
				'default_value'         => false,
				'exclude_default_token' => true,
			),
		);
	}

	/**
	 * Define tokens for this action
	 *
	 * @return array
	 */
	public function define_tokens() {
		return array(
			'PRODUCT_ID'    => array(
				'name' => esc_html_x( 'Product ID', 'WooCommerce', 'uncanny-automator-pro' ),
				'type' => 'int',
			),
			'PRODUCT_TITLE' => array(
				'name' => esc_html_x( 'Product title', 'WooCommerce', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'PRODUCT_SKU'   => array(
				'name' => esc_html_x( 'Product SKU', 'WooCommerce', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'PRODUCT_URL'   => array(
				'name' => esc_html_x( 'Product URL', 'WooCommerce', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
		);
	}

	/**
	 * Process the action
	 *
	 * @param int   $user_id
	 * @param array $action_data
	 * @param int   $recipe_id
	 * @param array $args
	 * @param array $parsed
	 *
	 * @return bool
	 */
	public function process_action( $user_id, $action_data, $recipe_id, $args, $parsed ) {
		// Validate required fields
		if ( empty( $parsed['WC_PRODUCT_TITLE'] ) ) {
			$this->add_log_error( esc_html_x( 'Product title is required', 'WooCommerce', 'uncanny-automator-pro' ) );
			return false;
		}

		if ( empty( $parsed['WC_PRODUCT_PRICE'] ) ) {
			$this->add_log_error( esc_html_x( 'Product price is required', 'WooCommerce', 'uncanny-automator-pro' ) );
			return false;
		}

		// Validate price is numeric
		if ( ! is_numeric( $parsed['WC_PRODUCT_PRICE'] ) ) {
			$this->add_log_error( esc_html_x( 'Product price must be a valid number', 'WooCommerce', 'uncanny-automator-pro' ) );
			return false;
		}

		// Validate sale price if provided
		if ( ! empty( $parsed['WC_PRODUCT_SALE_PRICE'] ) && ! is_numeric( $parsed['WC_PRODUCT_SALE_PRICE'] ) ) {
			$this->add_log_error( esc_html_x( 'Sale price must be a valid number', 'WooCommerce', 'uncanny-automator-pro' ) );
			return false;
		}

		// Set default values for required fields
		$product_status = ! empty( $parsed['WC_PRODUCT_STATUS'] ) ? sanitize_text_field( $parsed['WC_PRODUCT_STATUS'] ) : 'publish';

		// Create the product post
		$product_post = array(
			'post_type'    => 'product',
			'post_title'   => sanitize_text_field( $parsed['WC_PRODUCT_TITLE'] ),
			'post_name'    => sanitize_title( sanitize_text_field( $parsed['WC_PRODUCT_TITLE'] ) ),
			'post_status'  => $product_status,
			'post_content' => wp_kses_post( $parsed['WC_PRODUCT_DESCRIPTION'] ),
			'post_excerpt' => wp_kses_post( $parsed['WC_PRODUCT_SHORT_DESCRIPTION'] ),
		);

		$product_id = wp_insert_post( $product_post, true );

		if ( is_wp_error( $product_id ) ) {
			$error_message = $product_id->get_error_message();
			$error_code    = $product_id->get_error_code();
			$this->add_log_error(
				sprintf(
					// translators: %1$s: error code, %2$s: error message
					esc_html_x( 'Failed to create product (Error: %1$s): %2$s', 'WooCommerce', 'uncanny-automator-pro' ),
					$error_code,
					$error_message
				)
			);
			return false;
		}

		// Set product type to simple
		wp_set_object_terms( $product_id, 'simple', 'product_type' );

		// Set categories if provided
		if ( ! empty( $parsed['WC_PRODUCT_CATEGORIES'] ) ) {
			$product_categories = json_decode( $parsed['WC_PRODUCT_CATEGORIES'] );
			if ( ! empty( $product_categories ) ) {
				$product_categories = array_filter( array_map( 'intval', $product_categories ) );
				wp_set_object_terms( $product_id, $product_categories, 'product_cat' );
			}
		}

		// Set tags if provided
		if ( ! empty( $parsed['WC_PRODUCT_TAGS'] ) ) {
			$product_tags = json_decode( $parsed['WC_PRODUCT_TAGS'] );
			if ( ! empty( $product_tags ) ) {
				$product_tags = array_filter( array_map( 'intval', $product_tags ) );
				wp_set_object_terms( $product_id, $product_tags, 'product_tag' );
			}
		}

		// Set product meta data with proper defaults
		$product_meta_data = array(
			'_sku'           => sanitize_text_field( $parsed['WC_PRODUCT_SKU'] ),
			'_regular_price' => sanitize_text_field( $parsed['WC_PRODUCT_PRICE'] ),
			'_sale_price'    => sanitize_text_field( $parsed['WC_PRODUCT_SALE_PRICE'] ),
			'_manage_stock'  => wc_bool_to_string( $parsed['WC_MANAGE_STOCK'] ),
			'_stock_status'  => ! empty( $parsed['WC_PRODUCT_STOCK_STATUS'] ) ? sanitize_text_field( $parsed['WC_PRODUCT_STOCK_STATUS'] ) : 'instock',
			'_featured'      => wc_bool_to_string( $parsed['IS_FEATURED_PRODUCT'] ),
			'_visibility'    => ! empty( $parsed['WC_PRODUCT_CATALOG_VISIBILITY'] ) ? sanitize_text_field( $parsed['WC_PRODUCT_CATALOG_VISIBILITY'] ) : 'visible',
			'_virtual'       => wc_bool_to_string( $parsed['WC_PRODUCT_VIRTUAL'] ),
			'_downloadable'  => wc_bool_to_string( $parsed['WC_PRODUCT_DOWNLOADABLE'] ),
		);

		// Only set stock quantity if stock management is enabled
		if ( wc_string_to_bool( $parsed['WC_MANAGE_STOCK'] ) ) {
			$product_meta_data['_stock'] = sanitize_text_field( $parsed['WC_PRODUCT_STOCK_QUANTITY'] );
		}

		// Update product meta
		foreach ( $product_meta_data as $meta_key => $meta_value ) {
			// Always update meta values, even if they are 'no' for boolean fields
			update_post_meta( $product_id, $meta_key, $meta_value );
		}

		// Set the price (use sale price if available, otherwise regular price)
		$price = ! empty( $parsed['WC_PRODUCT_SALE_PRICE'] ) ? $parsed['WC_PRODUCT_SALE_PRICE'] : $parsed['WC_PRODUCT_PRICE'];
		update_post_meta( $product_id, '_price', $price );

		// Add product image if provided
		if ( ! empty( $parsed['WC_PRODUCT_IMAGE'] ) ) {
			$this->add_product_image( $parsed['WC_PRODUCT_IMAGE'], $product_id );
		}

		// Hydrate tokens
		$this->hydrate_tokens(
			array(
				'PRODUCT_ID'    => $product_id,
				'PRODUCT_TITLE' => $parsed['WC_PRODUCT_TITLE'],
				'PRODUCT_SKU'   => $parsed['WC_PRODUCT_SKU'],
				'PRODUCT_URL'   => get_permalink( $product_id ),
			)
		);

		return true;
	}

	/**
	 * Adds a featured image using the image URL and post ID.
	 *
	 * @param string $image_url
	 * @param int    $post_id
	 *
	 * @return int|string|\WP_Error
	 */
	private function add_product_image( $image_url, $post_id ) {
		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		// Prevents double image downloading.
		$existing_media_id = absint( attachment_url_to_postid( $image_url ) );

		// If existing_media_id is not equals 0, it means media already exists.
		if ( 0 !== $existing_media_id ) {
			// Overwrite the image url with the existing media.
			$image_url = $existing_media_id;
		}

		// Supports numeric input.
		if ( is_numeric( $image_url ) ) {
			// The $image_url is numeric.
			return set_post_thumbnail( $post_id, $image_url );
		}

		// Otherwise, download and store the image as attachment.
		$attachment_id = media_sideload_image( $image_url, $post_id, null, 'id' );

		// Assign the downloaded attachment ID to the post.
		set_post_thumbnail( $post_id, $attachment_id );

		if ( is_wp_error( $attachment_id ) ) {
			$error_message = $attachment_id->get_error_message();
			$error_code    = $attachment_id->get_error_code();
			$this->add_log_error(
				sprintf(
					// translators: %1$s: error code, %2$s: error message
					esc_html_x( 'Failed to add product image (Error: %1$s): %2$s', 'WooCommerce', 'uncanny-automator-pro' ),
					$error_code,
					$error_message
				)
			);
		}

		return $attachment_id;
	}
}
