<?php
namespace Uncanny_Automator_Pro\Integrations\Woocommerce_Subscription\Services;

/**
 * Order Resolver
 *
 * Resolves the most relevant order object to represent a subscription context.
 *
 * Resolution priority (from highest to lowest):
 *  1) Most recent FAILED renewal order
 *  2) Most recent renewal order
 *  3) Parent order (original purchase)
 *  4) Last order associated with the subscription (any type)
 *  5) The subscription object itself (WC_Subscription extends WC_Order)
 *
 * Why:
 * - Previously this resolver often returned the parent or a generic "last" order.
 *   That caused “Order ID” / “Order total” tokens to show the original purchase instead of
 *   the renewal that actually failed, which confused users.
 * - We now explicitly prefer the renewal order context, and specifically the most recent
 *   FAILED renewal when present. If no renewal orders exist, behavior falls back to the old
 *   logic so existing automations keep working.
 *
 * Implementation notes:
 * - Renewal orders are fetched via WC_Subscription::get_related_orders( 'ids', 'renewal' ).
 * - Orders are sorted by creation date (newest first); works with both CPT and HPOS.
 * - This is a light operation; renewal lists are typically small.
 *
 * @since 6.8.0 Enhanced order resolution to prioritize failed renewal orders.
 */
class Order_Resolver {

	/**
	 * The subscription whose related order we want to resolve.
	 *
	 * @var \WC_Subscription
	 */
	private $subscription;

	/**
	 * Order_Resolver constructor.
	 *
	 * @param \WC_Subscription $subscription A valid subscription instance.
	 *
	 * @throws \InvalidArgumentException If a non-subscription is passed.
	 * @since 6.8.0 Added validation for subscription parameter.
	 */
	public function __construct( $subscription ) {
		if ( ! is_a( $subscription, 'WC_Subscription' ) ) {
			throw new \InvalidArgumentException( 'Expected WC_Subscription object.' );
		}
		$this->subscription = $subscription;
	}

	/**
	 * Get the best order to represent this subscription context.
	 *
	 * Preference is given to the most relevant renewal order to ensure order tokens
	 * (e.g., Order ID, Order total) reflect the renewal attempt rather than the parent.
	 *
	 * @return \WC_Order|\WC_Subscription|null The resolved order (or the subscription as a fallback).
	 * @since 6.8.0 Enhanced to prioritize renewal orders over parent orders.
	 */
	public function get_order() {

		// 1) Renewal orders first (prefer a failed one, then most recent).
		$renewal = $this->get_latest_failed_or_latest_renewal_order();
		if ( $renewal ) {
			return $renewal;
		}

		// 2) Parent order (original purchase).
		$parent = $this->subscription->get_parent();
		if ( $parent ) {
			return $parent;
		}

		// 3) Last order associated with the subscription (any type).
		$last = $this->subscription->get_last_order();
		if ( $last ) {
			return $last;
		}

		// 4) Fallback: the subscription itself (it extends WC_Order).
		return $this->subscription;
	}

	/**
	 * Get the most relevant renewal order:
	 * - First, try the most recent FAILED renewal order.
	 * - If none are failed, return the most recent renewal order.
	 *
	 * @return \WC_Order|null The chosen renewal order or null if none exist.
	 * @since 6.8.0 Added method to resolve the most relevant renewal order.
	 */
	private function get_latest_failed_or_latest_renewal_order() {

		// get_related_orders('ids', 'renewal') returns a map of [order_id => order_id].
		$ids_map = $this->subscription->get_related_orders( 'ids', 'renewal' );
		if ( empty( $ids_map ) || ! is_array( $ids_map ) ) {
			return null;
		}

		// Load WC_Order objects for all renewal IDs.
		$orders = array_filter(
			array_map( 'wc_get_order', array_values( $ids_map ) ),
			static function ( $o ) {
				return $o instanceof \WC_Order;
			}
		);

		if ( empty( $orders ) ) {
			return null;
		}

		// Sort newest → oldest by creation date (compatible with CPT & HPOS).
		usort(
			$orders,
			static function ( $a, $b ) {
				$ta = $a->get_date_created() ? $a->get_date_created()->getTimestamp() : 0;
				$tb = $b->get_date_created() ? $b->get_date_created()->getTimestamp() : 0;
				return $tb <=> $ta;
			}
		);

		// Prefer the most recent failed renewal.
		foreach ( $orders as $o ) {
			if ( $o->has_status( 'failed' ) ) {
				return $o;
			}
		}

		// Otherwise, use the newest renewal order.
		return $orders[0] ?? null;
	}
}
