<?php

namespace Uncanny_Automator_Pro\Integrations\Woocommerce_Subscription\Actions;

use Exception;
use Uncanny_Automator\Recipe\Action;

/**
 * Class WC_SUBSCRIPTIONCANCEL
 * @package Uncanny_Automator_Pro
 */
class Cancel_Subscription_Variation_Non_Variable extends Action {

	/**
	 * Define and register the action by pushing it into the Automator object Cancel the user's subscription to {a product}
	 *
	 * @return void
	 */
	public function setup_action() {

		$this->set_action_code( 'WCSUBSCRIPTIONCANCELLED' );
		$this->set_action_meta( 'WOOSUBSCRIPTIONS' );
		$this->set_integration( 'WOOCOMMERCE_SUBSCRIPTION' );
		$this->set_is_pro( true );

		$this->set_sentence(
			sprintf(
				// translators: %1$s: Product, %2$s: Variation
				esc_attr_x(
					"Cancel the user's subscription to {{a variable subscription product:%1\$s}}",
					'WooCommerce Subscription',
					'uncanny-automator-pro'
				),
				$this->get_action_meta()
			)
		);

		$this->set_readable_sentence(
			esc_attr_x(
				"Cancel the user's subscription to {{a variable subscription product}}",
				'WooCommerce Subscription',
				'uncanny-automator-pro'
			)
		);
	}

	/**
	 * @return array
	 */
	public function options() {

		$product = array(
			'input_type'  => 'select',
			'option_code' => $this->get_action_meta(),
			'label'       => esc_attr_x( 'Subscription product', 'WooCommerce Subscription', 'uncanny-automator-pro' ),
			'required'    => true,
			'options'     => array(),
			'ajax'        => array(
				'endpoint' => 'automator_select_all_wc_subscriptions',
				'event'    => 'on_load',
			),
		);

		return array(
			$product,
		);
	}

	/**
	 * Validation function when the trigger action is hit
	 *
	 * Original logic from Woo - intact.
	 *
	 * @param $user_id
	 * @param $action_data
	 * @param $recipe_id
	 */
	public function process_action( $user_id, $action_data, $recipe_id, $args, $parsed ) {

		if ( ! function_exists( 'wcs_get_users_subscriptions' ) ) {
			throw new Exception( 'WooCommerce Subscriptions is not active' );
		}

		$subscriptions = wcs_get_users_subscriptions( $user_id );
		$product_id    = $parsed[ $this->action_meta ];

		if ( empty( $subscriptions ) ) {
			throw new Exception( esc_html_x( 'No subscription is associated with the user', 'WooCommerce Subscription', 'uncanny-automator-pro' ) );
		}

		$subscription_cancelled = false;
		$error_message          = esc_html_x( 'The user was not a subscriber of the specified product.', 'WooCommerce Subscription', 'uncanny-automator-pro' );

		foreach ( $subscriptions as $subscription ) {
			$items = $subscription->get_items();
			foreach ( $items as $index => $item ) {
				if ( absint( $item->get_product_id() ) === absint( $product_id ) ) {
					if ( $subscription->has_status( array( 'active' ) ) && $subscription->can_be_updated_to( 'cancelled' ) ) {
						$subscription->update_status( 'cancelled' );
						$subscription_cancelled = true;
					} else {
						$error_message = esc_html_x( 'We are not able to change subscription status.', 'WooCommerce Subscription', 'uncanny-automator-pro' );
					}
				}
			}
		}

		if ( false === $subscription_cancelled ) {
			throw new Exception( esc_html( $error_message ) );
		}

		return true;
	}
}
