<?php

namespace Uncanny_Automator_Pro\Loop_Filters;

use Uncanny_Automator_Pro\Loops\Filter\Base\Loop_Filter;

/**
 * Class WC_BOOKINGS_USER_PURCHASED_PRODUCT
 *
 * @package Uncanny_Automator_Pro
 */
class WC_BOOKINGS_USER_PURCHASED_PRODUCT extends Loop_Filter {

	/**
	 * @return void
	 * @throws \Exception
	 */
	public function setup() {
		$this->set_integration( 'WC_BOOKINGS' );
		$this->set_meta( 'WC_BOOKINGS_USER_PURCHASED_PRODUCT' );
		$this->set_sentence( esc_html_x( 'The user {{has/has not}} purchased {{a specific bookable product}}', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
		$this->set_sentence_readable(
			sprintf(
			// translators: %1$s: Criteria (has/has not), %2$s: Bookable product
				esc_html_x( 'The user {{has/has not:%1$s}} purchased {{a specific bookable product:%2$s}}', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				'CRITERIA',
				$this->get_meta()
			)
		);
		$this->set_fields( array( $this, 'load_options' ) );
		$this->set_entities( array( $this, 'retrieve_users_purchased_bookable_product' ) );
		$this->set_loop_type( 'users' );
	}

	/**
	 * @return mixed[]
	 */
	public function load_options() {
		return array(
			$this->get_meta() => array(
				array(
					'option_code'           => 'CRITERIA',
					'type'                  => 'select',
					'supports_custom_value' => false,
					'label'                 => esc_html_x( 'Criteria', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'options'               => array(
						array(
							'text'  => esc_html_x( 'has', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
							'value' => 'has',
						),
						array(
							'text'  => esc_html_x( 'has not', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
							'value' => 'has-not',
						),
					),
				),
				array(
					'option_code'           => $this->get_meta(),
					'type'                  => 'select',
					'label'                 => esc_html_x( 'Bookable product', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'options'               => array(),
					'supports_custom_value' => false,
					'ajax'                  => array(
						'endpoint' => 'fetch_bookable_products',
						'event'    => 'on_load',
					),
				),
			),
		);
	}


	/**
	 * @param array{WC_BOOKINGS_USER_PURCHASED_PRODUCT:string,CRITERIA:string} $fields
	 *
	 * @return int[]
	 */
	public function retrieve_users_purchased_bookable_product( $fields ) {
		$criteria   = $fields['CRITERIA'];
		$product_id = $fields['WC_BOOKINGS_USER_PURCHASED_PRODUCT'];

		if ( empty( $criteria ) || empty( $product_id ) ) {
			return array();
		}

		$user_ids = $this->get_users_with_bookable_product_orders( $product_id );
		$users    = $user_ids;

		if ( 'has-not' === $criteria ) {
			$all_user_ids = $this->get_all_user_ids();
			$users        = array_diff( $all_user_ids, $user_ids );
		}

		return ! empty( $users ) ? $users : array();
	}

	/**
	 * Get user IDs who have orders with bookable products
	 *
	 * @param string $product_id Product ID or '-1' for any bookable product
	 * @return array
	 */
	private function get_users_with_bookable_product_orders( $product_id ) {
		global $wpdb;

		// If -1 is selected, get all bookable product IDs
		if ( intval( '-1' ) === intval( $product_id ) ) {
			$bookable_product_ids = $this->get_all_bookable_product_ids();
			if ( empty( $bookable_product_ids ) ) {
				return array();
			}

			// Use IN statement for multiple products
			$placeholders = implode( ',', array_fill( 0, count( $bookable_product_ids ), '%s' ) );
			$query        = "
				SELECT DISTINCT oi.order_id
				FROM {$wpdb->prefix}woocommerce_order_itemmeta oim
				INNER JOIN {$wpdb->prefix}woocommerce_order_items oi ON oim.order_item_id = oi.order_item_id
				WHERE oim.meta_key = '_product_id'
				AND oim.meta_value IN ($placeholders)
			";
			$order_ids    = $wpdb->get_col( $wpdb->prepare( $query, ...$bookable_product_ids ) );
		} else {
			// Single product query
			$query     = "
				SELECT DISTINCT oi.order_id
				FROM {$wpdb->prefix}woocommerce_order_itemmeta oim
				INNER JOIN {$wpdb->prefix}woocommerce_order_items oi ON oim.order_item_id = oi.order_item_id
				WHERE oim.meta_key = '_product_id'
				AND oim.meta_value = %s
			";
			$order_ids = $wpdb->get_col( $wpdb->prepare( $query, $product_id ) );
		}

		$user_ids = array();
		foreach ( $order_ids as $order_id ) {
			$order = wc_get_order( $order_id );
			if ( $order ) {
				$user_id = $order->get_user_id();
				if ( $user_id ) {
					$user_ids[] = $user_id;
				}
			}
		}

		return array_unique( $user_ids );
	}

	/**
	 * Get all user IDs
	 *
	 * @return array
	 */
	private function get_all_user_ids() {
		$all_users = new \WP_User_Query(
			array(
				'cache_results' => false,
				'fields'        => 'ID',
			)
		);

		return $all_users->get_results();
	}

	/**
	 * Get all bookable product IDs
	 *
	 * @return array
	 */
	private function get_all_bookable_product_ids() {
		$args = array(
			'post_type'      => 'product',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'tax_query'      => array(
				array(
					'taxonomy' => 'product_type',
					'field'    => 'slug',
					'terms'    => 'booking',
				),
			),
		);

		return get_posts( $args );
	}
}
