<?php

namespace Uncanny_Automator_Pro\Integrations\WooCommerce_Bookings;

use Uncanny_Automator_Pro\Action_Condition;

/**
 * Class WC_BOOKINGS_DURATION
 *
 * @package Uncanny_Automator_Pro
 */
class WC_BOOKINGS_DURATION extends Action_Condition {

	/**
	 * Define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration = 'WC_BOOKINGS';
		$this->name        = esc_html_x( 'A booking duration {{meets a criteria}}', 'WooCommerce Bookings', 'uncanny-automator-pro' );
		$this->code        = 'DURATION';
		// translators: %1$s: Criteria (comparison), %2$s: Duration value, %3$s: Duration unit
		$this->dynamic_name  = sprintf(
			esc_html_x( 'A booking duration is {{criteria:%1$s}} {{a specific duration:%2$s}} {{unit:%3$s}}', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			'CRITERIA',
			'DURATION',
			'DURATION_UNIT'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * fields
	 *
	 * @return array
	 */
	public function fields() {

		return array(
			$this->field->text(
				array(
					'option_code' => 'BOOKING_ID',
					'label' => esc_html_x( 'Booking ID', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'text',
					'placeholder' => '',
					'description' => esc_html_x( 'Enter the booking ID or use a token', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'supports_tokens' => true,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'CRITERIA',
					'label' => esc_html_x( 'Comparison', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'greater_than',
							'text' => esc_html_x( 'greater than', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'greater_than_equal',
							'text' => esc_html_x( 'greater than or equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'less_than',
							'text' => esc_html_x( 'less than', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'less_than_equal',
							'text' => esc_html_x( 'less than or equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'equal',
							'text' => esc_html_x( 'equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'not_equal',
							'text' => esc_html_x( 'not equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
					'show_label_in_sentence' => false,
				)
			),
			$this->field->text(
				array(
					'option_code' => 'DURATION',
					'label' => esc_html_x( 'Duration', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'int',
					'description' => esc_html_x( 'Enter the duration value', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'show_label_in_sentence' => false,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'DURATION_UNIT',
					'label' => esc_html_x( 'Duration unit', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'minutes',
							'text' => esc_html_x( 'Minutes', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'hours',
							'text' => esc_html_x( 'Hours', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'days',
							'text' => esc_html_x( 'Days', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
					'default_value' => 'minutes',
					'show_label_in_sentence' => false,
				)
			),
		);
	}

	/**
	 * Evaluate_condition
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$booking_id    = $this->get_parsed_option( 'BOOKING_ID' );
		$criteria      = $this->get_parsed_option( 'CRITERIA' );
		$duration      = intval( $this->get_parsed_option( 'DURATION' ) );
		$duration_unit = $this->get_parsed_option( 'DURATION_UNIT' );

		// Validate booking ID
		if ( empty( $booking_id ) || ! is_numeric( $booking_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid booking ID provided.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the booking
		$booking = \get_wc_booking( $booking_id );
		if ( ! $booking ) {
			$this->condition_failed( esc_html_x( 'Booking not found.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get booking start and end times (these return timestamps)
		$start_timestamp = $booking->get_start();
		$end_timestamp   = $booking->get_end();

		if ( ! $start_timestamp || ! $end_timestamp ) {
			$this->condition_failed( esc_html_x( 'Booking has no valid start or end time.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Calculate booking duration in minutes using timestamps directly
		$booking_duration_minutes = ( $end_timestamp - $start_timestamp ) / 60;

		// Convert input duration to minutes based on selected unit
		$duration_in_minutes = $this->convert_duration_to_minutes( $duration, $duration_unit );

		// Evaluate the condition
		$condition_met = false;
		switch ( $criteria ) {
			case 'greater_than':
				$condition_met = ( $booking_duration_minutes > $duration_in_minutes );
				break;
			case 'greater_than_equal':
				$condition_met = ( $booking_duration_minutes >= $duration_in_minutes );
				break;
			case 'less_than':
				$condition_met = ( $booking_duration_minutes < $duration_in_minutes );
				break;
			case 'less_than_equal':
				$condition_met = ( $booking_duration_minutes <= $duration_in_minutes );
				break;
			case 'equal':
				$condition_met = ( $booking_duration_minutes == $duration_in_minutes );
				break;
			case 'not_equal':
				$condition_met = ( $booking_duration_minutes != $duration_in_minutes );
				break;
		}

		// If the condition is not met, send an error message
		if ( false === $condition_met ) {
			$message = $this->generate_error_message( $booking_duration_minutes, $duration, $duration_unit, $criteria );
			$this->condition_failed( $message );
		}
	}

	/**
	 * Convert duration to minutes based on unit
	 *
	 * @param int $duration
	 * @param string $unit
	 * @return int
	 */
	private function convert_duration_to_minutes( $duration, $unit ) {
		switch ( $unit ) {
			case 'hours':
				return $duration * 60;
			case 'days':
				return $duration * 24 * 60;
			case 'minutes':
			default:
				return $duration;
		}
	}

	/**
	 * Check if WooCommerce Bookings dependencies are active
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return class_exists( 'WooCommerce' ) && class_exists( 'WC_Bookings' );
	}

	/**
	 * Generate_error_message
	 *
	 * @param int $booking_duration_minutes
	 * @param int $expected_duration
	 * @param string $duration_unit
	 * @param string $criteria
	 *
	 * @return string
	 */
	public function generate_error_message( $booking_duration_minutes, $expected_duration, $duration_unit, $criteria ) {

		$booking_duration_formatted  = $this->format_duration( $booking_duration_minutes );
		$expected_duration_formatted = $this->format_duration_with_unit( $expected_duration, $duration_unit );

		$comparison_text = $this->get_comparison_text( $criteria );

		return sprintf(
			/* translators: 1: Actual booking duration, 2: Comparison text, 3: Expected duration */
			esc_html_x( 'Booking duration is %1$s but expected %2$s %3$s', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			$booking_duration_formatted,
			$comparison_text,
			$expected_duration_formatted
		);
	}

	/**
	 * Format duration for display
	 *
	 * @param float $minutes
	 * @return string
	 */
	private function format_duration( $minutes ) {
		$whole_hours       = floor( $minutes / 60 );
		$remaining_minutes = $minutes % 60;

		if ( $whole_hours > 0 && $remaining_minutes > 0 ) {
			// translators: %1$d: Hours, %2$d: Minutes
			return sprintf( esc_html_x( '%1$d hours %2$d minutes', 'WooCommerce Bookings', 'uncanny-automator-pro' ), $whole_hours, $remaining_minutes );
		} elseif ( $whole_hours > 0 ) {
			// translators: %d: Number of hours for booking duration display
			return sprintf( esc_html_x( '%d hours', 'WooCommerce Bookings', 'uncanny-automator-pro' ), $whole_hours );
		} else {
			// translators: %d: Number of minutes for booking duration display
			return sprintf( esc_html_x( '%d minutes', 'WooCommerce Bookings', 'uncanny-automator-pro' ), $remaining_minutes );
		}
	}

	/**
	 * Format duration with specific unit for display
	 *
	 * @param int $duration
	 * @param string $unit
	 * @return string
	 */
	private function format_duration_with_unit( $duration, $unit ) {
		switch ( $unit ) {
			case 'hours':
				// translators: %d: Number of hours for duration unit display
				return sprintf( esc_html_x( '%d hours', 'WooCommerce Bookings', 'uncanny-automator-pro' ), $duration );
			case 'days':
				// translators: %d: Number of days for duration unit display
				return sprintf( esc_html_x( '%d days', 'WooCommerce Bookings', 'uncanny-automator-pro' ), $duration );
			case 'minutes':
			default:
				// translators: %d: Number of minutes for duration unit display
				return sprintf( esc_html_x( '%d minutes', 'WooCommerce Bookings', 'uncanny-automator-pro' ), $duration );
		}
	}

	/**
	 * Get comparison text for error messages
	 *
	 * @param string $criteria
	 * @return string
	 */
	private function get_comparison_text( $criteria ) {
		switch ( $criteria ) {
			case 'greater_than':
				return esc_html_x( 'greater than', 'WooCommerce Bookings', 'uncanny-automator-pro' );
			case 'greater_than_equal':
				return esc_html_x( 'greater than or equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' );
			case 'less_than':
				return esc_html_x( 'less than', 'WooCommerce Bookings', 'uncanny-automator-pro' );
			case 'less_than_equal':
				return esc_html_x( 'less than or equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' );
			case 'equal':
				return esc_html_x( 'equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' );
			case 'not_equal':
				return esc_html_x( 'not equal to', 'WooCommerce Bookings', 'uncanny-automator-pro' );
			default:
				return esc_html_x( 'greater than', 'WooCommerce Bookings', 'uncanny-automator-pro' );
		}
	}
}

new WC_BOOKINGS_DURATION();
