<?php

namespace Uncanny_Automator_Pro\Integrations\WooCommerce_Bookings;

use Uncanny_Automator_Pro\Action_Condition;

/**
 * Class WC_BOOKINGS_CONTAINS_PRODUCT
 *
 * @package Uncanny_Automator_Pro
 */
class WC_BOOKINGS_CONTAINS_PRODUCT extends Action_Condition {

	/**
	 * Define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration = 'WC_BOOKINGS';
		$this->name        = esc_html_x( 'A booking contains {{a specific bookable product}}', 'WooCommerce Bookings', 'uncanny-automator-pro' );
		$this->code        = 'CONTAINS_PRODUCT';
		// translators: %1$s: Bookable product
		$this->dynamic_name  = sprintf(
			esc_html_x( 'A booking contains {{a specific bookable product:%1$s}}', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			'BOOKABLE_PRODUCT'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * fields
	 *
	 * @return array
	 */
	public function fields() {

		$products_field_args = array(
			'option_code' => 'BOOKABLE_PRODUCT',
			'label' => esc_html_x( 'Bookable product', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			'required' => true,
			'options' => $this->get_bookable_products_options(),
		);

		return array(
			$this->field->text(
				array(
					'option_code' => 'BOOKING',
					'label' => esc_html_x( 'Booking ID', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'text',
					'placeholder' => '',
					'description' => esc_html_x( 'Enter the booking ID or use a token', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'supports_tokens' => true,
				)
			),
			$this->field->select_field_args( $products_field_args ),
		);
	}

	/**
	 * Get bookable products options
	 *
	 * @return array
	 */
	public function get_bookable_products_options() {
		$args = array(
			'post_type' => 'product',
			'post_status' => 'publish',
			'orderby' => 'title',
			'order' => 'ASC',
			'posts_per_page' => 9999,
			'tax_query' => array(
				array(
					'taxonomy' => 'product_type',
					'field' => 'slug',
					'terms' => 'booking',
				),
			),
		);

		$bookable_products = get_posts( $args );
		$return            = array();

		foreach ( $bookable_products as $bookable_product ) {
			$return[] = array(
				'value' => $bookable_product->ID,
				'text' => $bookable_product->post_title,
			);
		}

		return $return;
	}

	/**
	 * Evaluate_condition
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$booking_id          = $this->get_parsed_option( 'BOOKING' );
		$bookable_product_id = $this->get_parsed_option( 'BOOKABLE_PRODUCT' );

		// Validate booking ID
		if ( empty( $booking_id ) || ! is_numeric( $booking_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid booking ID provided.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		if ( empty( $bookable_product_id ) || ! is_numeric( $bookable_product_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid bookable product ID provided.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the booking
		$booking = \get_wc_booking( $booking_id );
		if ( ! $booking ) {
			$this->condition_failed( esc_html_x( 'Booking not found.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the product ID from the booking
		$product_id = $booking->get_product_id();

		if ( ! $product_id ) {
			$this->condition_failed( esc_html_x( 'Booking has no associated product.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Check if the booking's product is the selected bookable product
		if ( $product_id != $bookable_product_id ) {
			$product_title = \get_the_title( $bookable_product_id );
			$message       = sprintf(
				/* translators: 1: Product title */
				esc_html_x( 'Booking product is not the bookable product "%1$s"', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$product_title
			);
			$this->condition_failed( $message );
		}

		// If we reach here, the condition is met (booking contains the selected bookable product)
		// No action needed - the condition passes
	}

	/**
	 * Check if WooCommerce Bookings dependencies are active
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return class_exists( 'WooCommerce' ) && class_exists( 'WC_Bookings' );
	}
}

new WC_BOOKINGS_CONTAINS_PRODUCT();
