<?php

namespace Uncanny_Automator_Pro\Integrations\URL;

/**
 * Class URL_Helpers
 *
 * @package Uncanny_Automator
 */
class URL_Pro_Helpers extends \Uncanny_Automator\Integrations\URL\URL_Helpers {


	/**
	 * URL_HAS_PARAM_VALUE & URL_HAS_PARAM_VALUE_LOGGED_IN options
	 *
	 * @return array
	 */
	public function url_has_param_value_get_options() {
		return array(
			array(
				'option_code'              => 'URL_CONDITION',
				'label'                    => esc_html_x( 'Condition', 'URL', 'uncanny-automator-pro' ),
				'description'              => esc_html_x( 'Whether any or all of the following parameter values match', 'URL', 'uncanny-automator-pro' ),
				'input_type'               => 'select',
				'required'                 => true,
				'options_show_id'          => false,
				'supports_multiple_values' => false,
				'supports_custom_value'    => false,
				'relevant_tokens'          => array(),
				'options'                  => array(
					array(
						'value' => '-1',
						'text'  => esc_html_x( 'Any', 'URL', 'uncanny-automator-pro' ),
					),
					array(
						'value' => 'all',
						'text'  => esc_html_x( 'All', 'URL', 'uncanny-automator-pro' ),
					),
				),
			),
			array(
				'option_code'       => 'URL_PARAMETERS',
				'label'             => esc_html_x( 'Parameters and values', 'URL', 'uncanny-automator-pro' ),
				'input_type'        => 'repeater',
				'required'          => true,
				'relevant_tokens'   => array(),
				'fields'            => array(
					array(
						'option_code'           => 'PARAM_NAME',
						'label'                 => esc_html_x( 'Parameter name', 'URL', 'uncanny-automator-pro' ),
						'input_type'            => 'text',
						'required'              => true,
						'supports_tokens'       => true,
						'supports_custom_value' => true,
					),
					array(
						'option_code'           => 'PARAM_VALUE',
						'label'                 => esc_html_x( 'Parameter value', 'URL', 'uncanny-automator-pro' ),
						'input_type'            => 'text',
						'required'              => true,
						'supports_tokens'       => true,
						'supports_custom_value' => true,
					),
				),
				'add_row_button'    => esc_html_x( 'Add parameter', 'URL', 'uncanny-automator-pro' ),
				'remove_row_button' => esc_html_x( 'Remove parameter', 'URL', 'uncanny-automator-pro' ),
			),
		);
	}

	/**
	 * URL_HAS_PARAM_VALUE & URL_HAS_PARAM_VALUE_LOGGED_IN validation
	 *
	 * This method validates whether the current request contains URL parameters
	 * that match the configured trigger conditions.
	 *
	 * @param array $trigger_data The trigger configuration data
	 * @param array $hook_args The hook arguments containing request data
	 *
	 * @return bool True if validation passes, false otherwise
	 */
	public function url_has_param_value_validate_trigger( $trigger_data, $hook_args ) {
		// Early validation: ensure required trigger meta fields exist
		if ( empty( $trigger_data['meta']['URL_CONDITION'] ) || empty( $trigger_data['meta']['URL_PARAMETERS'] ) ) {
			return false;
		}

		// Get request data, fallback to GET and POST superglobals securely
		// We use FILTER_UNSAFE_RAW to preserve legitimate URL parameter values
		$request_data = $hook_args['request'] ?? array();
		$get_params   = $request_data['get'] ?? filter_input_array( INPUT_GET, FILTER_UNSAFE_RAW ) ?? array();
		$post_params  = $request_data['post'] ?? filter_input_array( INPUT_POST, FILTER_UNSAFE_RAW ) ?? array();

		// Extract and sanitize the trigger configuration
		$condition  = sanitize_text_field( $trigger_data['meta']['URL_CONDITION'] );
		$parameters = json_decode( $trigger_data['meta']['URL_PARAMETERS'], true ) ?? array();

		// Validate that parameters array exists and is properly formatted
		if ( empty( $parameters ) || false === is_array( $parameters ) ) {
			return false;
		}

		// Counter to track how many parameters match the conditions
		$found_params = 0;

		// Loop through each configured parameter to check for matches
		foreach ( $parameters as $param ) {
			// Sanitize parameter name and value for safe comparison
			$name  = sanitize_text_field( $param['PARAM_NAME'] ?? '' );
			$value = sanitize_text_field( $param['PARAM_VALUE'] ?? '' );

			// Only process parameters with valid names and values
			if ( '' !== $name && '' !== $value ) {
				// Check GET parameters first (URL query string)
				if ( isset( $get_params[ $name ] ) && $get_params[ $name ] === $value ) {
					++$found_params;
					continue; // Skip POST check since we found a match
				}

				// Check POST parameters if GET didn't match
				if ( isset( $post_params[ $name ] ) && $post_params[ $name ] === $value ) {
					++$found_params;
					continue;
				}
			}
		}

		// Return validation result based on condition type:
		// -1 (Any): Return true if at least one parameter matches (0 < $found_params)
		// all: Return true only if ALL parameters match ($found_params === count($parameters))
		return intval( '-1' ) === intval( $condition ) ? 0 < $found_params : ( 'all' === $condition && count( $parameters ) === $found_params );
	}

	/**
	 * URL_HAS_PARAM_VALUE & URL_HAS_PARAM_VALUE_LOGGED_IN hydrate tokens
	 *
	 * @param $trigger_data
	 * @param $hook_args
	 *
	 * @return array
	 */
	public function url_has_param_value_hydrate_tokens( $trigger_data, $hook_args ) {
		$trigger = array(
			'meta'      => $trigger_data['meta'],
			'hook_args' => $hook_args,
		);

		return array(
			'URL'                 => $this->get_token_value( 'URL', array( $trigger ) ),
			'ALL_REPEATER_PARAMS' => $this->get_token_value( 'ALL_REPEATER_PARAMS', array( $trigger ) ),
			'ALL_POST_PARAMS'     => $this->get_token_value( 'ALL_POST_PARAMS', array( $trigger ) ),
			'ALL_GET_PARAMS'      => $this->get_token_value( 'ALL_GET_PARAMS', array( $trigger ) ),
		);
	}
}
