<?php

namespace Uncanny_Automator_Pro\Integrations\SureCart;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class SURECART_USERS_SUBSCRIPTION_RENEWED
 *
 * @package Uncanny_Automator_Pro
 * @method \Uncanny_Automator_Pro\Integrations\SureCart\SureCart_Pro_Helpers get_item_helpers()
 */
class SURECART_USERS_SUBSCRIPTION_RENEWED extends Trigger {

	/**
	 * Constant TRIGGER_CODE.
	 *
	 * @var string
	 */
	const TRIGGER_CODE = 'USER_SUBSCRIPTION_RENEWED';

	/**
	 * Constant TRIGGER_META.
	 *
	 * @var string
	 */
	const TRIGGER_META = 'PRODUCT';

	/**
	 * Setup trigger
	 *
	 * @return void
	 */
	protected function setup_trigger() {
		$this->set_integration( 'SURECART' );
		$this->set_trigger_code( self::TRIGGER_CODE );
		$this->set_trigger_meta( self::TRIGGER_META );
		$this->set_is_pro( true );
		$this->set_is_login_required( false );
		$this->set_trigger_type( 'user' );
		$this->add_action( 'surecart/subscription_renewed', 10, 2 );

		// translators: %1$s: SureCart Product
		$this->set_sentence( sprintf( esc_html_x( 'A user renews a subscription to {{a product:%1$s}}', 'SureCart', 'uncanny-automator' ), $this->get_trigger_meta() ) );
		$this->set_readable_sentence( esc_html_x( 'A user renews a subscription to {{a product}}', 'SureCart', 'uncanny-automator' ) );
	}

	/**
	 * Loads available options for the Trigger.
	 *
	 * @return array The available trigger options.
	 */
	public function options() {
		return array(
			array(
				'option_code' => $this->get_trigger_meta(),
				'label'       => esc_html_x( 'Product', 'SureCart', 'uncanny-automator' ),
				'input_type'  => 'select',
				'required'    => true,
				'options'     => $this->get_item_helpers()->get_products_dropdown_options(),
				'relevant_tokens' => array(
					$this->get_trigger_meta() => esc_html_x( 'Selected product', 'SureCart', 'uncanny-automator' ),
				),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger The trigger data.
	 * @param array $hook_args The hook arguments.
	 *
	 * @return bool True if validation was successful.
	 */
	public function validate( $trigger, $hook_args ) {
		list( $subscription, $event ) = $hook_args;

		// Find the right WP user
		$user_id = $this->get_item_helpers()->get_user_id_from_customer( $subscription->customer ?? null );
		$this->set_user_id( $user_id );

		// Check the subscription product
		$product_id = $trigger['meta'][ $this->get_trigger_meta() ];

		if ( intval( '-1' ) === intval( $product_id ) ) {
			return true;
		}

		/** @var \SureCart\Models\Purchase $purchase */
		$purchase = class_exists( 'SureCart\Models\Purchase' ) ? \SureCart\Models\Purchase::with( array( 'purchase.product' ) )->find( $subscription->purchase ) : null;

		if ( $purchase && (string) $purchase->product->id === (string) $product_id ) {
			return true;
		}

		return false;
	}

	/**
	 * Hydrate tokens with values.
	 *
	 * @param array $trigger The trigger data.
	 * @param array $hook_args The hook arguments.
	 *
	 * @return array The token values.
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		list( $subscription, $event ) = $hook_args;

		$tokens = array();

		/** @var \SureCart\Models\Subscription $subscription_info */
		$subscription_info = class_exists( 'SureCart\Models\Subscription' ) ? \SureCart\Models\Subscription::with( array( 'purchase', 'purchase.initial_order', 'order.checkout', 'purchase.product', 'price', 'payment_method' ) )->find( $subscription->id ) : null;

		if ( $subscription_info ) {
			// Use existing token hydration methods
			$surecart_tokens = new SureCart_Pro_Tokens_New_Framework();

			// Get common tokens
			$common_tokens = $surecart_tokens->hydrate_common_tokens();
			$tokens        = array_merge( $tokens, $common_tokens );

			// Get subscription tokens
			$subscription_tokens = $surecart_tokens->hydrate_subscription_tokens( $subscription_info );
			$tokens              = array_merge( $tokens, $subscription_tokens );

			// Add specific product name for this trigger
			$tokens['PRODUCT'] = $subscription_info->purchase->product->name;
		}

		return $tokens;
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger The trigger configuration.
	 * @param array $tokens The existing tokens.
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		// Use existing token definitions
		$surecart_tokens = new SureCart_Pro_Tokens_New_Framework();

		$custom_tokens = array_merge(
			$surecart_tokens->common_tokens(),
			$surecart_tokens->subscription_tokens(),
			$surecart_tokens->billing_tokens(),
			array(
				array(
					'tokenId'   => 'PRODUCT',
					'tokenName' => esc_html_x( 'Product', 'SureCart', 'uncanny-automator' ),
					'tokenType' => 'text',
				),
			)
		);

		return array_merge( $tokens, $custom_tokens );
	}
}
