<?php

namespace Uncanny_Automator_Pro\Integrations\SureCart;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class SURECART_ANON_REFUND_ISSUED
 *
 * @package Uncanny_Automator_Pro
 * @method \Uncanny_Automator_Pro\Integrations\SureCart\SureCart_Pro_Helpers get_item_helpers()
 */
class SURECART_ANON_REFUND_ISSUED extends Trigger {

	/**
	 * Constant TRIGGER_CODE.
	 *
	 * @var string
	 */
	const TRIGGER_CODE = 'ANON_REFUND_ISSUED';

	/**
	 * Constant TRIGGER_META.
	 *
	 * @var string
	 */
	const TRIGGER_META = 'PRODUCT';

	/**
	 * Setup trigger
	 *
	 * @return void
	 */
	protected function setup_trigger() {
		$this->set_integration( 'SURECART' );
		$this->set_trigger_code( self::TRIGGER_CODE );
		$this->set_trigger_meta( self::TRIGGER_META );
		$this->set_is_pro( true );
		$this->set_is_login_required( false );
		$this->set_trigger_type( 'anonymous' );
		$this->add_action( 'surecart/refund_succeeded', 10, 2 );

		// translators: %1$s: SureCart Product
		$this->set_sentence( sprintf( esc_html_x( 'A refund for {{a product:%1$s}} is issued to a customer', 'SureCart', 'uncanny-automator' ), $this->get_trigger_meta() ) );
		$this->set_readable_sentence( esc_html_x( 'A refund for {{a product}} is issued to a customer', 'SureCart', 'uncanny-automator' ) );
	}

	/**
	 * Loads available options for the Trigger.
	 *
	 * @return array The available trigger options.
	 */
	public function options() {
		return array(
			array(
				'option_code' => $this->get_trigger_meta(),
				'label'       => esc_html_x( 'Product', 'SureCart', 'uncanny-automator' ),
				'input_type'  => 'select',
				'required'    => true,
				'options'     => $this->get_item_helpers()->get_products_dropdown_options(),
				'relevant_tokens' => array(
					$this->get_trigger_meta() => esc_html_x( 'Selected product', 'SureCart', 'uncanny-automator' ),
				),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger The trigger data.
	 * @param array $hook_args The hook arguments.
	 *
	 * @return bool True if validation was successful.
	 */
	public function validate( $trigger, $hook_args ) {
		$product_id     = $trigger['meta'][ $this->get_trigger_meta() ];
		list( $refund ) = $hook_args;

		/** @var \SureCart\Models\Charge $charge */
		$charge = class_exists( 'SureCart\Models\Charge' ) ? \SureCart\Models\Charge::with( array( 'checkout', 'checkout.order', 'checkout.purchases', 'purchase.product', 'checkout.customer' ) )->find( $refund->charge ) : null;

		if ( ! $charge || ! $charge->checkout ) {
			return false;
		}

		$checkout = $charge->checkout;

		// Extract user ID from checkout customer field
		$user_id = $this->get_item_helpers()->get_user_id_from_customer( $checkout->customer ?? null );
		$this->set_user_id( $user_id );

		if ( intval( '-1' ) === intval( $product_id ) ) {
			return true;
		}

		foreach ( $checkout->purchases->data as $purchase_data ) {
			$product = $purchase_data->product;

			if ( (string) $product->id === (string) $product_id ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Hydrate tokens with values.
	 *
	 * @param array $trigger The trigger data.
	 * @param array $hook_args The hook arguments.
	 *
	 * @return array The token values.
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		list( $refund ) = $hook_args;

		/** @var \SureCart\Models\Charge $charge */
		$charge = class_exists( 'SureCart\Models\Charge' ) ? \SureCart\Models\Charge::with( array( 'checkout', 'checkout.order', 'checkout.purchases', 'purchase.product', 'checkout.customer' ) )->find( $refund->charge ) : null;

		$tokens = array();

		if ( $charge && $charge->checkout ) {
			$checkout = $charge->checkout;

			// Use existing token hydration methods
			$surecart_tokens = new SureCart_Pro_Tokens_New_Framework();

			// Get common tokens
			$common_tokens = $surecart_tokens->hydrate_common_tokens();
			$tokens        = array_merge( $tokens, $common_tokens );

			// Get order tokens
			$order_tokens = $surecart_tokens->hydrate_order_tokens( $checkout );
			$tokens       = array_merge( $tokens, $order_tokens );

			// Get billing tokens
			$billing_tokens = $surecart_tokens->hydrate_billing_tokens( $checkout );
			$tokens         = array_merge( $tokens, $billing_tokens );

			// Add product information for this specific trigger
			$product_ids = array();
			foreach ( $checkout->purchases->data as $purchase_data ) {
				$product       = $purchase_data->product;
				$product_ids[] = $product->id;
			}
			$tokens['PRODUCT'] = implode( ', ', $product_ids );
		}

		return $tokens;
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger The trigger configuration.
	 * @param array $tokens The existing tokens.
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		// Use existing token definitions
		$surecart_tokens = new SureCart_Pro_Tokens_New_Framework();

		$custom_tokens = array_merge(
			$surecart_tokens->common_tokens(),
			$surecart_tokens->order_tokens(),
			$surecart_tokens->billing_tokens(),
			array(
				array(
					'tokenId'   => 'PRODUCT',
					'tokenName' => esc_html_x( 'Product', 'SureCart', 'uncanny-automator' ),
					'tokenType' => 'text',
				),
			)
		);

		return array_merge( $tokens, $custom_tokens );
	}
}
