<?php

namespace Uncanny_Automator_Pro\Integrations\Presto;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class PRESTO_VIDEOPERCENT
 *
 * @package Uncanny_Automator_Pro
 * @method \Uncanny_Automator\Integrations\Presto\Presto_Helpers get_item_helpers()
 */
class PRESTO_VIDEOPERCENT extends Trigger {
	/**
	 * Setup trigger
	 *
	 * @return void
	 */
	protected function setup_trigger() {
		$this->set_integration( 'PRESTO' );
		$this->set_trigger_code( 'PRESTOVIDEOPERCENT' );
		$this->set_trigger_meta( 'PRESTOVIDEO' );
		$this->set_is_pro( true );

		// Hook into video progress event
		$this->add_action( 'presto_player_progress' );

		$this->set_action_args_count( 2 );

		// translators: %1$s: is a percentage, %2$s: is a video
		$this->set_sentence( sprintf( esc_html_x( 'A user watches at least {{a specific percentage:%1$s}} of {{a video:%2$s}}', 'Presto', 'uncanny-automator-pro' ), 'PERCENTAGE:' . $this->get_trigger_meta(), $this->get_trigger_meta() ) );

		$this->set_readable_sentence(
			esc_html_x( 'A user watches at least {{a specific percentage}} of {{a video}}', 'Presto', 'uncanny-automator-pro' )
		);
	}

	/**
	 * Loads available options for the Trigger.
	 *
	 * @return array The available trigger options.
	 */
	public function options() {
		// phpcs:ignore WordPress.Arrays.MultipleStatementAlignment.DoubleArrowNotAligned -- Array alignment varies for readability
		return array(
			array(
				'option_code'     => $this->get_trigger_meta(),
				'label'           => esc_html_x( 'Video', 'Presto', 'uncanny-automator-pro' ),
				'input_type'      => 'select',
				'required'        => true,
				'options'         => $this->get_item_helpers()->get_all_presto_videos( true ),
				'relevant_tokens' => array(),
			),
			array(
				'option_code'     => 'PERCENTAGE',
				'label'           => esc_html_x( 'Percentage', 'Presto', 'uncanny-automator-pro' ),
				'input_type'      => 'select',
				'required'        => true,
				'options'         => array(
					array(
						'value' => '_10',
						'text'  => esc_html_x( '10%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_20',
						'text'  => esc_html_x( '20%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_30',
						'text'  => esc_html_x( '30%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_40',
						'text'  => esc_html_x( '40%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_50',
						'text'  => esc_html_x( '50%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_60',
						'text'  => esc_html_x( '60%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_70',
						'text'  => esc_html_x( '70%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_80',
						'text'  => esc_html_x( '80%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_90',
						'text'  => esc_html_x( '90%', 'Presto', 'uncanny-automator-pro' ),
					),
					array(
						'value' => '_100',
						'text'  => esc_html_x( '100%', 'Presto', 'uncanny-automator-pro' ),
					),
				),
				'relevant_tokens' => array(),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger The trigger data.
	 * @param array $hook_args The hook arguments.
	 *
	 * @return bool True if validation was successful.
	 */
	public function validate( $trigger, $hook_args ) {
		list( $video_id, $percent ) = $hook_args;

		$selected_video_id = $trigger['meta'][ $this->get_trigger_meta() ];
		$selected_percent  = $trigger['meta']['PERCENTAGE'];

		// If "Any video" is selected (-1), return true
		if ( intval( '-1' ) === intval( $selected_video_id ) ) {
			return true;
		}

		// Check if the video matches the selected video
		$normalized_video_id = $this->get_item_helpers()->get_normalized_video_id( $video_id );
		if ( ! $normalized_video_id ) {
			return false;
		}

		if ( absint( $selected_video_id ) !== absint( $normalized_video_id ) ) {
			return false;
		}

		// Check if the percentage matches the selected percentage
		$expected_percent = str_replace( '_', '', $selected_percent );
		return intval( $percent ) >= intval( $expected_percent );
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger The trigger configuration.
	 * @param array $hook_args The hook arguments.
	 *
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		list( $video_id, $percent ) = $hook_args;

		$video = $this->get_item_helpers()->normalize_video_data( $video_id );
		if ( ! $video ) {
			return array();
		}

		return array(
			$this->get_trigger_meta()                 => $video->title,
			$this->get_trigger_meta() . '_ID'         => $video->id,
			$this->get_trigger_meta() . '_POST_TITLE' => ! empty( $video->hub_id ) ? get_the_title( $video->hub_id ) : '-',
			'PERCENTAGE'                              => $percent . '%',
		);
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger The trigger configuration.
	 * @param array $tokens The existing tokens.
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		return array(
			$this->get_trigger_meta()                 => array(
				'name'      => esc_html_x( 'Video title', 'Presto', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => $this->get_trigger_meta(),
				'tokenName' => esc_html_x( 'Video title', 'Presto', 'uncanny-automator-pro' ),
			),
			$this->get_trigger_meta() . '_ID'         => array(
				'name'      => esc_html_x( 'Video ID', 'Presto', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => $this->get_trigger_meta() . '_ID',
				'tokenName' => esc_html_x( 'Video ID', 'Presto', 'uncanny-automator-pro' ),
			),
			$this->get_trigger_meta() . '_POST_TITLE' => array(
				'name'      => esc_html_x( 'Media hub title', 'Presto', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => $this->get_trigger_meta() . '_POST_TITLE',
				'tokenName' => esc_html_x( 'Media hub title', 'Presto', 'uncanny-automator-pro' ),
			),
			'PERCENTAGE'                              => array(
				'name'      => esc_html_x( 'Percentage watched', 'Presto', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'PERCENTAGE',
				'tokenName' => esc_html_x( 'Percentage watched', 'Presto', 'uncanny-automator-pro' ),
			),
		);
	}
}
