<?php

namespace Uncanny_Automator_Pro;

/**
 * Class LD_USERCOURSEACCESSEXPIRED
 *
 * @package Uncanny_Automator_Pro
 */
class LD_USERCOURSEACCESSEXPIRED {

	/**
	 * Integration code
	 *
	 * @var string
	 */
	public static $integration = 'LD';

	private $trigger_code;
	private $trigger_meta;

	/**
	 * Set up Automator trigger constructor.
	 */
	public function __construct() {
		$this->trigger_code = 'LD_COURSEACCESSEXPIRED';
		$this->trigger_meta = 'LDCOURSE';
		$this->define_trigger();
	}

	/**
	 * Define and register the trigger by pushing it into the Automator object
	 */
	public function define_trigger() {
		$trigger = array(
			'author'              => Automator()->get_author_name( $this->trigger_code ),
			'support_link'        => Automator()->get_author_support_link( $this->trigger_code, 'integration/learndash/' ),
			'is_pro'              => true,
			'integration'         => self::$integration,
			'code'                => $this->trigger_code,
			/* translators: Logged-in trigger - LearnDash */
			'sentence'            => sprintf( esc_attr__( "A user's access to {{a course:%1\$s}} expires", 'uncanny-automator-pro' ), $this->trigger_meta ),
			/* translators: Logged-in trigger - LearnDash */
			'select_option_name'  => esc_attr__( "A user's access to {{a course}} expires", 'uncanny-automator-pro' ),
			'action'              => 'learndash_user_course_access_expired',
			'priority'            => 10,
			'accepted_args'       => 2,
			'validation_function' => array(
				$this,
				'user_course_access_expired',
			),
			'options_callback'    => array( $this, 'load_options' ),
		);

		Automator()->register->trigger( $trigger );
	}

	/**
	 * @return array[]
	 */
	public function load_options() {
		return Automator()->utilities->keep_order_of_options(
			array(
				'options' => array(
					Automator()->helpers->recipe->learndash->options->all_ld_courses( null, $this->trigger_meta ),
				),
			)
		);
	}

	/**
	 * Validation function when the trigger action is hit
	 *
	 * @param $user_id
	 * @param $course_id
	 */
	public function user_course_access_expired( $user_id, $course_id ) {
		// Validate input parameters
		if ( empty( $user_id ) || empty( $course_id ) ) {
			return;
		}

		// Get the actual course expiry timestamp from LearnDash
		$course_expiry_timestamp = ld_course_access_expires_on( $course_id, $user_id );
		
		if ( empty( $course_expiry_timestamp )  ) {
			// No expiry set, shouldn't happen but safety check
			return;
		}

		// Check if this specific expiry has already been processed
		if ( $this->is_course_expiry_already_processed( $user_id, $course_id, $course_expiry_timestamp ) ) {
			return;
		}

		// Mark this specific expiry event as processed
		$this->mark_course_expiry_as_processed( $user_id, $course_id, $course_expiry_timestamp );

		$args = array(
			'code'         => $this->trigger_code,
			'meta'         => $this->trigger_meta,
			'post_id'      => $course_id,
			'user_id'      => $user_id,
			'is_signed_in' => true,
		);

		Automator()->maybe_add_trigger_entry( $args );
	}

	/**
	 * Get the course expiry key
	 * @param mixed $course_id
	 * @param mixed $expiry_timestamp
	 * @return string
	 */
	private function get_course_expiry_key( $course_id, $expiry_timestamp ) {
		// Short key: courseID_timestamp
		return sprintf( '%d_%d', $course_id, (int) $expiry_timestamp );
	}

	/**
	 * Check if this course expiry is already processed for the user
	 */
	private function is_course_expiry_already_processed( $user_id, $course_id, $expiry_timestamp ): bool {
		$user_processed_expiries = get_user_meta( $user_id, 'uap_ld_processed_expiries', true );
		
		if ( empty( $user_processed_expiries ) || ! is_array( $user_processed_expiries ) ) {
			return false;
		}
		
		$course_key = $this->get_course_expiry_key( $course_id, $expiry_timestamp );
		return isset( $user_processed_expiries[ $course_key ] );
	}

	/**
	 * Mark the course expiry as processed for the user
	 */
	private function mark_course_expiry_as_processed( $user_id, $course_id, $expiry_timestamp ): void {
		$user_processed_expiries = get_user_meta( $user_id, 'uap_ld_processed_expiries', true );
		
		if ( empty( $user_processed_expiries ) || ! is_array( $user_processed_expiries ) ) {
			$user_processed_expiries = array();
		}
		
		$course_key = $this->get_course_expiry_key( $course_id, $expiry_timestamp );
		$user_processed_expiries[ $course_key ] = time();
		
		update_user_meta( $user_id, 'uap_ld_processed_expiries', $user_processed_expiries );
	}

	/**
	 * Clear processed expiries for a specific user/course combination
	 * This can be useful for testing or if course access is restored
	 *
	 * @param int $user_id   User ID
	 * @param int $course_id Course ID (optional - if not provided, clears all for user)
	 */
	public static function clear_processed_expiries( $user_id, $course_id = null ) {
		if ( $course_id ) {
			// Clear specific course for user
			$user_processed_expiries = get_user_meta( $user_id, 'uap_ld_processed_expiries', true );
			
			if ( ! empty( $user_processed_expiries ) && is_array( $user_processed_expiries ) ) {
				// Remove all entries for this course
				foreach ( $user_processed_expiries as $key => $timestamp ) {
					if ( strpos( $key, $course_id . '_' ) === 0 ) {
						unset( $user_processed_expiries[ $key ] );
					}
				}
				update_user_meta( $user_id, 'uap_ld_processed_expiries', $user_processed_expiries );
			}
		} else {
			// Clear all expiries for user
			delete_user_meta( $user_id, 'uap_ld_processed_expiries' );
		}
	}

	/**
	 * Clear all processed expiries for all users
	 * This can be useful for bulk cleanup
	 */
	public static function clear_all_processed_expiries() {
		global $wpdb;
		
		// Delete all user meta entries that match our pattern
		$wpdb->query( 
			$wpdb->prepare( 
				"DELETE FROM {$wpdb->usermeta} WHERE meta_key = %s",
				'uap_ld_processed_expiries'
			) 
		);
	}

}
