<?php

namespace Uncanny_Automator_Pro\Integrations\Github;

/**
 * GitHub Pro Tokens Class`
 *
 * Provides common token definitions and hydration methods for GitHub triggers
 * to eliminate code duplication across trigger files.
 *
 * @package Uncanny_Automator_Pro\Integrations\Github\Tokens
 */
class GitHub_Pro_Tokens {

	/**
	 * Static property to store webhooks helper reference.
	 *
	 * @var object|null
	 */
	private static $webhooks = null;

	/**
	 * Set the webhooks helper reference for use in token value methods.
	 *
	 * @param object $webhooks Webhooks helper instance.
	 * @return void
	 */
	public static function set_webhooks( $webhooks ) {
		self::$webhooks = $webhooks;
	}

	/**
	 * Get repository token definitions.
	 *
	 * @return array
	 */
	public static function get_repository_token_definitions() {
		return array(
			array(
				'tokenId'   => 'GITHUB_REPO_ID',
				'tokenName' => esc_html_x( 'Repository ID', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'GITHUB_REPO_NAME',
				'tokenName' => esc_html_x( 'Repository name', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_REPO_URL',
				'tokenName' => esc_html_x( 'Repository URL', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Hydrate repository token values.
	 *
	 * @param array $payload GitHub webhook payload.
	 *
	 * @return array
	 */
	public static function hydrate_repository_tokens( $payload ) {
		return array(
			'GITHUB_REPO_ID'   => self::get_value( $payload, 'repository.id' ),
			'GITHUB_REPO_NAME' => self::get_value( $payload, 'repository.name' ),
			'GITHUB_REPO_URL'  => self::get_value( $payload, 'repository.html_url' ),
		);
	}

	/**
	 * Get sender token definitions
	 *
	 * @return array
	 */
	public static function get_sender_token_definitions() {
		return array(
			array(
				'tokenId'   => 'GITHUB_SENDER_LOGIN',
				'tokenName' => esc_html_x( 'Sender login', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_SENDER_AVATAR_URL',
				'tokenName' => esc_html_x( 'Sender avatar URL', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_SENDER_EMAIL',
				'tokenName' => esc_html_x( 'Sender email', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Get sender token values
	 *
	 * @param array $payload GitHub webhook payload
	 * @return array
	 */
	public static function hydrate_sender_tokens( $payload ) {
		return array(
			'GITHUB_SENDER_LOGIN'      => self::get_value( $payload, 'sender.login' ),
			'GITHUB_SENDER_AVATAR_URL' => self::get_value( $payload, 'sender.avatar_url' ),
			'GITHUB_SENDER_EMAIL'      => self::get_sender_email( $payload ),
		);
	}

	/**
	 * Get sender email with multiple fallback strategies
	 *
	 * GitHub webhook payloads often don't include email addresses for privacy reasons.
	 * This method attempts multiple fallback strategies to find an email address.
	 *
	 * @param array $payload GitHub webhook payload
	 * @return string
	 */
	private static function get_sender_email( $payload ) {
		// Define email paths to check in order of preference
		$email_paths = array(
			'sender.email',             // Most common for some events
			'pusher.email',             // Push events
			'comment.user.email',       // Comment events
			'issue.user.email',         // Issue events
			'pull_request.user.email',  // PR events
			'release.author.email',     // Release events
			'forkee.owner.email',       // Fork events
			'deployment.creator.email', // Deployment events
			'workflow_run.actor.email',  // Workflow run events
		);

		foreach ( $email_paths as $path ) {
			$email = self::get_value( $payload, $path );
			if ( ! empty( $email ) ) {
				return $email;
			}
		}

		/**
		 * Filter to allow custom email extraction logic
		 *
		 * This filter allows advanced users to implement their own email
		 * extraction strategies, such as API calls, database lookups, or
		 * custom mapping logic.
		 *
		 * @param string $email The email address found (empty if none found)
		 * @param array  $payload The full GitHub webhook payload
		 *
		 * @return string The email address to use.
		 */
		$email = apply_filters( 'automator_github_sender_email', '', $payload );

		// If no email found, return empty string
		return $email;
	}

	/**
	 * Get pull request token definitions.
	 *
	 * @return array
	 */
	public static function get_pull_request_token_definitions() {
		return array(
			array(
				'tokenId'   => 'GITHUB_PULL_REQUEST_TITLE',
				'tokenName' => esc_html_x( 'Pull request title', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_PULL_REQUEST_BODY',
				'tokenName' => esc_html_x( 'Pull request body', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_PULL_REQUEST_BODY_RICH',
				'tokenName' => esc_html_x( 'Pull request body (formatted)', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_PULL_REQUEST_URL',
				'tokenName' => esc_html_x( 'Pull request URL', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_PULL_REQUEST_ID',
				'tokenName' => esc_html_x( 'Pull request ID', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_REF',
				'tokenName' => esc_html_x( 'Reference', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Hydrate pull request token values.
	 *
	 * @param array $payload GitHub webhook payload
	 * @return array
	 */
	public static function hydrate_pull_request_tokens( $payload ) {
		return array(
			'GITHUB_PULL_REQUEST_TITLE'     => self::get_value( $payload, 'pull_request.title' ),
			'GITHUB_PULL_REQUEST_BODY'      => self::get_value( $payload, 'pull_request.body' ),
			'GITHUB_PULL_REQUEST_URL'       => self::get_value( $payload, 'pull_request.html_url' ),
			'GITHUB_PULL_REQUEST_BODY_RICH' => self::$webhooks
				? self::$webhooks->get_rich_text_value( $payload, 'pull_request.body' )
				: '',
			'GITHUB_PULL_REQUEST_ID'        => self::get_value( $payload, 'pull_request.number' ),
			'GITHUB_REF'                    => self::get_value( $payload, 'pull_request.base.ref' ),
		);
	}

	/**
	 * Get release token definitions.
	 *
	 * @return array
	 */
	public static function get_release_token_definitions() {
		return array(
			array(
				'tokenId'   => 'GITHUB_RELEASE_TITLE',
				'tokenName' => esc_html_x( 'Release title', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_RELEASE_BODY',
				'tokenName' => esc_html_x( 'Release body', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_RELEASE_BODY_RICH',
				'tokenName' => esc_html_x( 'Release body (formatted)', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_RELEASE_URL',
				'tokenName' => esc_html_x( 'Release URL', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_RELEASE_TAG',
				'tokenName' => esc_html_x( 'Release tag', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Hydrate release token values.
	 *
	 * @param array $payload GitHub webhook payload
	 * @return array
	 */
	public static function hydrate_release_tokens( $payload ) {
		return array(
			'GITHUB_RELEASE_TITLE'     => self::get_value( $payload, 'release.name' ),
			'GITHUB_RELEASE_BODY'      => self::get_value( $payload, 'release.body' ),
			'GITHUB_RELEASE_URL'       => self::get_value( $payload, 'release.html_url' ),
			'GITHUB_RELEASE_BODY_RICH' => self::$webhooks
				? self::$webhooks->get_rich_text_value( $payload, 'release.body' )
				: '',
			'GITHUB_RELEASE_TAG'       => self::get_value( $payload, 'release.tag_name' ),
		);
	}

	/**
	 * Get issue token definitions.
	 *
	 * @return array
	 */
	public static function get_issue_token_definitions() {
		return array(
			array(
				'tokenId'   => 'GITHUB_ISSUE_TITLE',
				'tokenName' => esc_html_x( 'Issue title', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_ISSUE_BODY',
				'tokenName' => esc_html_x( 'Issue body', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_ISSUE_BODY_RICH',
				'tokenName' => esc_html_x( 'Issue body (formatted)', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_ISSUE_URL',
				'tokenName' => esc_html_x( 'Issue URL', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_ISSUE_NUMBER',
				'tokenName' => esc_html_x( 'Issue number', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'GITHUB_ISSUE_STATE',
				'tokenName' => esc_html_x( 'Issue state', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'GITHUB_ISSUE_CREATOR',
				'tokenName' => esc_html_x( 'Issue creator', 'GitHub', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Hydrate issue token values.
	 *
	 * @param array $payload GitHub webhook payload
	 * @return array
	 */
	public static function hydrate_issue_tokens( $payload ) {
		return array(
			'GITHUB_ISSUE_TITLE'     => self::get_value( $payload, 'issue.title' ),
			'GITHUB_ISSUE_BODY'      => self::get_value( $payload, 'issue.body' ),
			'GITHUB_ISSUE_URL'       => self::get_value( $payload, 'issue.html_url' ),
			'GITHUB_ISSUE_BODY_RICH' => self::$webhooks
				? self::$webhooks->get_rich_text_value( $payload, 'issue.body' )
				: '',
			'GITHUB_ISSUE_NUMBER'    => self::get_value( $payload, 'issue.number' ),
			'GITHUB_ISSUE_STATE'     => self::get_value( $payload, 'issue.state' ),
			'GITHUB_ISSUE_CREATOR'   => self::get_value( $payload, 'issue.user.login' ),
		);
	}

	/**
	 * Helper method to safely get payload values with webhooks check.
	 *
	 * @param array $payload GitHub webhook payload.
	 * @param string $key Dot notation key (e.g., 'repository.id').
	 * @return string
	 */
	private static function get_value( $payload, $key ) {
		if ( ! self::$webhooks ) {
			return '';
		}
		return self::$webhooks->get_payload_value( $payload, $key );
	}

	/**
	 * Add class identifier to tokens
	 *
	 * @param array $tokens The tokens
	 * @param string $identifier The identifier
	 * @return array
	 */
	public static function add_class_identifier( $tokens, $identifier ) {
		return array_map(
			function ( $token ) use ( $identifier ) {
				$token['tokenIdentifier'] = $identifier;
				return $token;
			},
			$tokens
		);
	}
}
