<?php
namespace Uncanny_Automator_Pro;

use FluentCrm\App\Models\Lists as FluentCrm_List;

/**
 * Class FCRM_USER_LIST_NOT_FOUND
 *
 * @package Uncanny_Automator_Pro
 */
class FCRM_USER_LIST_NOT_FOUND extends Action_Condition {

	/**
	 * Define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration = 'FCRM';

		/*translators: Token */
		$this->name = esc_html_x( 'A user does not belong to {{a list}}', 'Fluent Crm', 'uncanny-automator-pro' );

		$this->code = 'USER_LIST_NOT_FOUND';

		// translators: A token matches a value
		$this->dynamic_name = sprintf(
			// translators: %1$s: list
			esc_html_x( 'A user does not belong to {{a list:%1$s}}', 'Fluent Crm', 'uncanny-automator-pro' ),
			'LIST'
		);

		$this->is_pro = true;

		$this->requires_user = true;
	}

	/**
	 * Fields
	 *
	 * @return array
	 */
	public function fields() {

		$fcrm_lists_fields_args = array(
			'option_code'           => 'LIST',
			'label'                 => esc_html_x( 'List', 'Fluent Crm', 'uncanny-automator-pro' ),
			'required'              => true,
			'options'               => $this->fcrm_lists_options(),
			'supports_custom_value' => true,
		);

		return array(
			// List field
			$this->field->select_field_args( $fcrm_lists_fields_args ),
		);
	}

	/**
	 * Load options
	 *
	 * @return array[]
	 */
	public function fcrm_lists_options() {

		$options = array();

		if ( ! class_exists( 'FluentCrm\App\Models\Lists' ) ) {

			return array();

		}

		$lists = FluentCrm_List::orderBy( 'title', 'DESC' )->get();

		if ( ! empty( $lists ) ) {

			foreach ( $lists as $list ) {

				$options[] = array(
					'value' => $list->id,
					'text'  => $list->title,
				);

			}
		}

		return $options;
	}

	/**
	 * Method evaluate_condition.
	 *
	 * Sets the condition to fail with error message if the user belongs to the list.
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$list = $this->get_parsed_option( 'LIST' );

		if ( $this->fcrm_user_has_lists( $this->user_id, array( $list ) ) ) {

			$error_message = sprintf( 'The user with id: [%d] already belongs to list: [%s]', $this->user_id, $this->get_option( 'LIST_readable' ) );

			$this->condition_failed( $error_message ); // Dont translate.

		}
	}

	/**
	 * Method fcrm_get_user
	 *
	 * Get the Fluent CRM contact.
	 *
	 * @param int $user_id The wp user id of the Fluent CRM contact.
	 *
	 * @return \FluentCrmApi The Fluent CRM contact.
	 */
	public function fcrm_get_user( $user_id = 0 ) {

		if ( ! function_exists( 'FluentCrmApi' ) ) {

			return false;

		}

		$fcrm_contact = FluentCrmApi( 'contacts' );

		$current_contact = $fcrm_contact->getInstance()->where( 'user_id', $user_id )->first();

		return $current_contact;
	}

	/**
	 * Method fcrm_user_has_lists
	 *
	 * Check if the user belongs to any of the specified lists.
	 *
	 * @param int $user_id The wp user id of the Fluent CRM contact.
	 * @param array $lists The list ids (e.g. [1,2,5])
	 *
	 * @return boolean True if the user belongs to any of the lists. Otherwise false.
	 */
	public function fcrm_user_has_lists( $user_id = 0, $lists = array() ) {

		$contact = $this->fcrm_get_user( $user_id );

		if ( ! $contact ) {
			return false;
		}

		$user_list_ids = array();
		foreach ( $contact->lists as $list ) {
			$user_list_ids[] = $list->id;
		}

		// Check if any of the specified lists are in the user's lists
		return ! empty( array_intersect( $lists, $user_list_ids ) );
	}
}
