<?php

namespace Uncanny_Automator_Pro\Integrations\Easy_Digital_Downloads;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class EDD_USER_SUBSCRIBE_PRODWITHPRICEOPTION
 *
 * @package Uncanny_Automator
 */
class EDD_USER_SUBSCRIBE_PRODWITHPRICEOPTION extends Trigger {

	/**
	 * Trigger code.
	 *
	 * @var string
	 */
	protected $trigger_code = 'EDD_USER_SUBSCRIBE_PRODWITHPRICEOPTION';

	/**
	 * Trigger meta.
	 *
	 * @var string
	 */
	protected $trigger_meta = 'EDDPRODUCT';

	/**
	 * Price option meta.
	 *
	 * @var string
	 */
	protected $price_option_meta = 'EDDPRICEOPTION';

	/**
	 * Check if the trigger requirements are met.
	 *
	 * @return bool
	 */
	public function requirements_met() {
		if ( ! class_exists( 'EDD_Recurring' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Set up Automator trigger.
	 */
	protected function setup_trigger() {
		$this->set_integration( 'EDD' );
		$this->set_trigger_code( $this->trigger_code );
		$this->set_trigger_meta( $this->trigger_meta );
		$this->set_is_pro( true );
		$this->set_trigger_type( 'user' );
		$this->set_is_login_required( false );
		$this->set_support_link( Automator()->get_author_support_link( $this->trigger_code, 'integration/easy-digital-downloads/' ) );
		$this->set_sentence(
			sprintf(
				// translators: %1$s: download, %2$s: price option
				esc_html_x( 'A user subscribes to {{a price option:%2$s}} of {{a download:%1$s}}', 'Easy Digital Downloads', 'uncanny-automator' ),
				$this->get_trigger_meta(),
				$this->price_option_meta . ':' . $this->get_trigger_meta()
			)
		);
		$this->set_readable_sentence( esc_html_x( 'A user subscribes to {{a price option}} of {{a download}}', 'Easy Digital Downloads', 'uncanny-automator' ) );
		$this->add_action( 'edd_recurring_post_record_signup', 10, 3 );
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			$this->get_item_helpers()->all_edd_downloads( esc_html_x( 'Download', 'Easy Digital Downloads', 'uncanny-automator' ), $this->get_trigger_meta(), true, false ),
			array(
				'option_code'     => $this->price_option_meta,
				'label'           => esc_html_x( 'Price option', 'Easy Digital Downloads', 'uncanny-automator' ),
				'input_type'      => 'select',
				'required'        => true,
				'options'         => array(),
				'relevant_tokens' => array(),
				'ajax'            => array(
					'event'         => 'parent_fields_change',
					'endpoint'      => 'automator_edd_price_options_handler',
					'listen_fields' => array( $this->get_trigger_meta() ),
				),
			),
		);
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger
	 * @param array $tokens
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		return array(
			'EDD_DOWNLOAD_TITLE'              => array(
				'name'      => esc_html_x( 'Download title', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_DOWNLOAD_TITLE',
				'tokenName' => esc_html_x( 'Download title', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_DOWNLOAD_ID'                 => array(
				'name'      => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_ID',
				'tokenName' => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_DOWNLOAD_URL'                => array(
				'name'      => esc_html_x( 'Download URL', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_URL',
				'tokenName' => esc_html_x( 'Download URL', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => array(
				'name'      => esc_html_x( 'Download featured image ID', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_ID',
				'tokenName' => esc_html_x( 'Download featured image ID', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => array(
				'name'      => esc_html_x( 'Download featured image URL', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_URL',
				'tokenName' => esc_html_x( 'Download featured image URL', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_PRICE_OPTION_NAME'           => array(
				'name'      => esc_html_x( 'Price option name', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PRICE_OPTION_NAME',
				'tokenName' => esc_html_x( 'Price option name', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_PRICE_OPTION_ID'             => array(
				'name'      => esc_html_x( 'Price option ID', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_PRICE_OPTION_ID',
				'tokenName' => esc_html_x( 'Price option ID', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_ORDER_ID'                    => array(
				'name'      => esc_html_x( 'Order ID', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_ORDER_ID',
				'tokenName' => esc_html_x( 'Order ID', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_ORDER_URL'                   => array(
				'name'      => esc_html_x( 'Order URL (Admin)', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_ORDER_URL',
				'tokenName' => esc_html_x( 'Order URL (Admin)', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_ORDER_URL_USER'              => array(
				'name'      => esc_html_x( 'Order URL (User)', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_ORDER_URL_USER',
				'tokenName' => esc_html_x( 'Order URL (User)', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_ORDER_DISCOUNTS'             => array(
				'name'      => esc_html_x( 'Order discounts', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_DISCOUNTS',
				'tokenName' => esc_html_x( 'Order discounts', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_ORDER_SUBTOTAL'              => array(
				'name'      => esc_html_x( 'Order subtotal', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_SUBTOTAL',
				'tokenName' => esc_html_x( 'Order subtotal', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_ORDER_TAX'                   => array(
				'name'      => esc_html_x( 'Order tax', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_TAX',
				'tokenName' => esc_html_x( 'Order tax', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_ORDER_TOTAL'                 => array(
				'name'      => esc_html_x( 'Order total', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_TOTAL',
				'tokenName' => esc_html_x( 'Order total', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_PAYMENT_METHOD'              => array(
				'name'      => esc_html_x( 'Payment method', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PAYMENT_METHOD',
				'tokenName' => esc_html_x( 'Payment method', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_DISCOUNT_CODES'              => array(
				'name'      => esc_html_x( 'Discount codes used', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_DISCOUNT_CODES',
				'tokenName' => esc_html_x( 'Discount codes used', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_LICENSE_KEY'                 => array(
				'name'      => esc_html_x( 'License key', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_LICENSE_KEY',
				'tokenName' => esc_html_x( 'License key', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_RECURRING_PERIOD'            => array(
				'name'      => esc_html_x( 'Recurring period', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_RECURRING_PERIOD',
				'tokenName' => esc_html_x( 'Recurring period', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_SIGNUP_FEE'                  => array(
				'name'      => esc_html_x( 'Signup fee', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_SIGNUP_FEE',
				'tokenName' => esc_html_x( 'Signup fee', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_TIMES'                       => array(
				'name'      => esc_html_x( 'Times', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_TIMES',
				'tokenName' => esc_html_x( 'Times', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_FREE_TRIAL_PERIOD'           => array(
				'name'      => esc_html_x( 'Free trial period', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_FREE_TRIAL_PERIOD',
				'tokenName' => esc_html_x( 'Free trial period', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_SUBSCRIPTION_ID'             => array(
				'name'      => esc_html_x( 'Subscription ID', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_SUBSCRIPTION_ID',
				'tokenName' => esc_html_x( 'Subscription ID', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_SUBSCRIPTION_URL'            => array(
				'name'      => esc_html_x( 'Subscription URL (Admin)', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_SUBSCRIPTION_URL',
				'tokenName' => esc_html_x( 'Subscription URL (Admin)', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_CUSTOMER_ID'                 => array(
				'name'      => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_ID',
				'tokenName' => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_CUSTOMER_EMAIL'              => array(
				'name'      => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_EMAIL',
				'tokenName' => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_CUSTOMER_NAME'               => array(
				'name'      => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_NAME',
				'tokenName' => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_CUSTOMER_STATUS'             => array(
				'name'      => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_STATUS',
				'tokenName' => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
			'EDD_CUSTOMER_USER_ID'            => array(
				'name'      => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_USER_ID',
				'tokenName' => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator' ),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( empty( $hook_args ) || count( $hook_args ) < 2 ) {
			return false;
		}

		if ( ! isset( $trigger['meta'][ $this->trigger_meta ] ) ) {
			return false;
		}

		$selected_download     = $trigger['meta'][ $this->trigger_meta ];
		$selected_price_option = $trigger['meta'][ $this->price_option_meta ];

		// Fallback: try to get price option from different possible keys.
		if ( empty( $selected_price_option ) ) {
			if ( isset( $trigger['meta']['EDDPRICEOPTION'] ) ) {
				$selected_price_option = $trigger['meta']['EDDPRICEOPTION'];
			} elseif ( isset( $trigger['meta'][ 'EDDPRICEOPTION:' . $this->get_trigger_meta() ] ) ) {
				$selected_price_option = $trigger['meta'][ 'EDDPRICEOPTION:' . $this->get_trigger_meta() ];
			}
		}

		$subscription_object = $hook_args[0];
		$subscription        = $hook_args[1];
		$download_id         = $subscription['id'];
		$price_option_id     = isset( $subscription['price_id'] ) ? $subscription['price_id'] : null;

		// Set user ID for the trigger from customer data
		$user_id = 0;
		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = \edd_get_customer( $subscription_object->customer_id );
			if ( $customer && ! empty( $customer->user_id ) ) {
				$user_id = $customer->user_id;
			}
		}

		// Fallback to current user ID if no customer user ID found
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( $user_id ) {
			$this->set_user_id( $user_id );
		}

		// Check if download matches (allow "Any download" option (-1) or specific download).
		$download_matches = ( intval( '-1' ) === intval( $selected_download ) || absint( $selected_download ) === absint( $download_id ) );

		// Check if price option matches (allow "Any price option" option (-1) or specific price option).
		$price_option_matches = ( intval( '-1' ) === intval( $selected_price_option ) || absint( $selected_price_option ) === absint( $price_option_id ) );

		// Both download and price option must match.
		return $download_matches && $price_option_matches;
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		if ( empty( $hook_args ) || count( $hook_args ) < 2 ) {
			return array();
		}

		list( $subscription_object, $subscription, $payment_object ) = $hook_args;
		$purchase_data = $payment_object->purchase_data;

		$download_id     = $subscription['id'];
		$price_option_id = isset( $subscription['price_id'] ) ? $subscription['price_id'] : null;

		// Get subscription ID from subscription object.
		$subscription_id = '';
		if ( method_exists( $subscription_object, 'get_id' ) ) {
			$subscription_id = $subscription_object->get_id();
		} elseif ( isset( $subscription_object->id ) ) {
			$subscription_id = $subscription_object->id;
		}

		$parent_payment_id = '';
		if ( method_exists( $subscription_object, 'get_parent_payment_id' ) ) {
			$parent_payment_id = $subscription_object->get_parent_payment_id();
		} elseif ( isset( $subscription_object->parent_payment_id ) ) {
			$parent_payment_id = $subscription_object->parent_payment_id;
		}

		$download_post = get_post( $download_id );

		// Get price option name.
		$price_option_name = '';
		if ( null !== $price_option_id ) {
			$price_option_name = \edd_get_price_option_name( $download_id, $price_option_id );
		}

		// Get featured image.
		$featured_image_id  = get_post_thumbnail_id( $download_id );
		$featured_image_url = $featured_image_id ? wp_get_attachment_url( $featured_image_id ) : '';

		// Get order URL (Admin).
		$order_url = $parent_payment_id ? admin_url( 'edit.php?post_type=download&page=edd-payment-history&view=view-order-details&id=' . $parent_payment_id ) : '';

		// Get order URL (User) - receipt page.
		$order_url_user = $parent_payment_id ? \edd_get_receipt_page_uri( $parent_payment_id ) : '';

		// Get subscription URL (Admin).
		$subscription_url = $subscription_id ? admin_url( 'edit.php?post_type=download&page=edd-subscriptions&view=view-subscription&id=' . $subscription_id ) : '';

		// Get recurring period from subscription array.
		$recurring_period = '';
		if ( isset( $subscription['period'] ) ) {
			$recurring_period = $subscription['period'];
		} elseif ( method_exists( $subscription_object, 'get_period' ) ) {
			$recurring_period = $subscription_object->get_period();
		}

		// Get signup fee from subscription array.
		$signup_fee = '';
		if ( isset( $subscription['signup_fee'] ) ) {
			$signup_fee = number_format( $subscription['signup_fee'], 2 );
		} elseif ( method_exists( $subscription_object, 'get_signup_fee' ) ) {
			$signup_fee = number_format( $subscription_object->get_signup_fee(), 2 );
		}

		// Get times from subscription array.
		$times = '';
		if ( isset( $subscription['frequency'] ) ) {
			$times = $subscription['frequency'];
		} elseif ( method_exists( $subscription_object, 'get_frequency' ) ) {
			$times = $subscription_object->get_frequency();
		}

		// Get free trial period from subscription array.
		$free_trial_period = '';
		if ( isset( $subscription['trial_unit'] ) && isset( $subscription['trial_quantity'] ) ) {
			$free_trial_period = $subscription['trial_quantity'] . ' ' . $subscription['trial_unit'];
		} elseif ( method_exists( $subscription_object, 'get_trial_period' ) ) {
			$free_trial_period = $subscription_object->get_trial_period();
		}

		// Get customer information.
		$customer_id      = '';
		$customer_email   = '';
		$customer_name    = '';
		$customer_status  = '';
		$customer_user_id = '';

		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = \edd_get_customer( $subscription_object->customer_id );
			if ( $customer ) {
				$customer_id      = $customer->id;
				$customer_email   = $customer->email;
				$customer_name    = $customer->name;
				$customer_status  = $customer->status;
				$customer_user_id = $customer->user_id;
			}
		}

		$tokens = array(
			'EDD_DOWNLOAD_TITLE'              => $download_post ? $download_post->post_title : '',
			'EDD_DOWNLOAD_ID'                 => $download_id,
			'EDD_DOWNLOAD_URL'                => get_permalink( $download_id ),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => $featured_image_id,
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => $featured_image_url,
			'EDD_PRICE_OPTION_NAME'           => $price_option_name,
			'EDD_PRICE_OPTION_ID'             => $price_option_id,
			'EDD_ORDER_ID'                    => $parent_payment_id,
			'EDD_ORDER_URL'                   => $order_url,
			'EDD_ORDER_URL_USER'              => $order_url_user,
			'EDD_ORDER_DISCOUNTS'             => isset( $purchase_data['discount'] ) ? number_format( $purchase_data['discount'], 2 ) : '',
			'EDD_ORDER_SUBTOTAL'              => isset( $purchase_data['subtotal'] ) ? number_format( $purchase_data['subtotal'], 2 ) : '',
			'EDD_ORDER_TAX'                   => isset( $purchase_data['tax'] ) ? number_format( $purchase_data['tax'], 2 ) : '',
			'EDD_ORDER_TOTAL'                 => isset( $purchase_data['price'] ) ? number_format( $purchase_data['price'], 2 ) : '',
			'EDD_PAYMENT_METHOD'              => $parent_payment_id ? \edd_get_payment_gateway( $parent_payment_id ) : '',
			'EDD_DISCOUNT_CODES'              => isset( $purchase_data['user_info']['discount'] ) ? $purchase_data['user_info']['discount'] : '',
			'EDD_LICENSE_KEY'                 => $parent_payment_id ? $this->get_item_helpers()->get_licenses( $parent_payment_id ) : '',
			'EDD_RECURRING_PERIOD'            => $recurring_period,
			'EDD_SIGNUP_FEE'                  => $signup_fee,
			'EDD_TIMES'                       => $times,
			'EDD_FREE_TRIAL_PERIOD'           => $free_trial_period,
			'EDD_SUBSCRIPTION_ID'             => $subscription_id,
			'EDD_SUBSCRIPTION_URL'            => $subscription_url,
			'EDD_CUSTOMER_ID'                 => $customer_id,
			'EDD_CUSTOMER_EMAIL'              => $customer_email,
			'EDD_CUSTOMER_NAME'               => $customer_name,
			'EDD_CUSTOMER_STATUS'             => $customer_status,
			'EDD_CUSTOMER_USER_ID'            => $customer_user_id,
		);

		return $tokens;
	}
}
