<?php

namespace Uncanny_Automator_Pro\Integrations\Easy_Digital_Downloads;

use EDD_Payment;
use Uncanny_Automator\Recipe\Trigger;

/**
 * Class EDD_PAYMENT_FAILS
 *
 * @pacakge Uncanny_Automator_Pro
 */
class EDD_PAYMENT_FAILS extends Trigger {

	/**
	 * @return mixed|void
	 */
	protected function setup_trigger() {
		$this->set_integration( 'EDD' );
		$this->set_trigger_code( 'EDD_PAYMENT_FAILS' );
		$this->set_trigger_meta( 'EDD_PAYMENTS' );
		$this->set_is_pro( true );
		$this->set_sentence( esc_html_x( 'A payment fails', 'Easy Digital Downloads', 'uncanny-automator-pro' ) );
		$this->set_readable_sentence( esc_html_x( 'A payment fails', 'Easy Digital Downloads', 'uncanny-automator-pro' ) );
		$this->add_action( 'edd_before_payment_status_change', 99, 3 );
	}

	/**
	 * @param $trigger
	 * @param $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( ! isset( $hook_args[0], $hook_args[1] ) ) {
			return false;
		}

		if ( ! is_numeric( $hook_args[0] ) ) {
			return false;
		}

		$status = $hook_args[1];

		if ( 'failed' !== $status ) {
			return false;
		}

		// Set user ID for the trigger from payment data
		$payment_id = $hook_args[0];
		$payment    = new EDD_Payment( $payment_id );

		$user_id = 0;
		if ( ! empty( $payment->customer_id ) ) {
			$customer = edd_get_customer( $payment->customer_id );
			if ( $customer && ! empty( $customer->user_id ) ) {
				$user_id = $customer->user_id;
			}
		}

		// Fallback to current user ID if no customer user ID found
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( $user_id ) {
			$this->set_user_id( $user_id );
		}

		return true;
	}

	/**
	 * @param $trigger
	 * @param $tokens
	 *
	 * @return array|array[]
	 */
	public function define_tokens( $trigger, $tokens ) {
		$trigger_tokens = array(
			array(
				'tokenId'   => 'DOWNLOAD_ID',
				'tokenName' => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'DOWNLOAD_NAME',
				'tokenName' => esc_html_x( 'Download name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'DOWNLOAD_QTY',
				'tokenName' => esc_html_x( 'Download quantity', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'DOWNLOAD_SUBTOTAL',
				'tokenName' => esc_html_x( 'Download subtotal', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'DOWNLOAD_TAX',
				'tokenName' => esc_html_x( 'Download tax', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'DOWNLOAD_PRICE',
				'tokenName' => esc_html_x( 'Download price', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'PAYMENT_GATEWAY',
				'tokenName' => esc_html_x( 'Payment gateway', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'PAYMENT_CURRENCY',
				'tokenName' => esc_html_x( 'Payment currency', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_ID',
				'tokenName' => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_EMAIL',
				'tokenName' => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_NAME',
				'tokenName' => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_STATUS',
				'tokenName' => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_USER_ID',
				'tokenName' => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
		);

		return array_merge( $tokens, $trigger_tokens );
	}

	/**
	 * hydrate_tokens
	 *
	 * @param $trigger
	 * @param $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		$payment_id = $hook_args[0];
		$payment    = new EDD_Payment( $payment_id );

		if ( ! $payment instanceof EDD_Payment ) {
			return array(
				'DOWNLOAD_NAME'     => '',
				'DOWNLOAD_ID'       => '',
				'DOWNLOAD_QTY'      => '',
				'DOWNLOAD_SUBTOTAL' => '',
				'DOWNLOAD_TAX'      => '',
				'DOWNLOAD_PRICE'    => '',
				'PAYMENT_GATEWAY'   => '',
				'PAYMENT_CURRENCY'  => '',
			);
		}

		$cart_items = edd_get_payment_meta_cart_details( $payment_id );
		$item       = array_shift( $cart_items );

		// Get customer information.
		$customer_id      = '';
		$customer_email   = '';
		$customer_name    = '';
		$customer_status  = '';
		$customer_user_id = '';

		if ( ! empty( $payment->customer_id ) ) {
			$customer = edd_get_customer( $payment->customer_id );
			if ( $customer ) {
				$customer_id      = $customer->id;
				$customer_email   = $customer->email;
				$customer_name    = $customer->name;
				$customer_status  = $customer->status;
				$customer_user_id = $customer->user_id;
			}
		}

		return array(
			'DOWNLOAD_NAME'        => $item['name'],
			'DOWNLOAD_ID'          => $item['id'],
			'DOWNLOAD_QTY'         => $item['quantity'],
			'DOWNLOAD_SUBTOTAL'    => edd_currency_filter( edd_format_amount( $item['subtotal'] ) ),
			'DOWNLOAD_TAX'         => edd_currency_filter( edd_format_amount( $item['tax'] ) ),
			'DOWNLOAD_PRICE'       => edd_currency_filter( edd_format_amount( $item['price'] ) ),
			'PAYMENT_GATEWAY'      => $payment->gateway,
			'PAYMENT_CURRENCY'     => $payment->currency,
			'EDD_CUSTOMER_ID'      => $customer_id,
			'EDD_CUSTOMER_EMAIL'   => $customer_email,
			'EDD_CUSTOMER_NAME'    => $customer_name,
			'EDD_CUSTOMER_STATUS'  => $customer_status,
			'EDD_CUSTOMER_USER_ID' => $customer_user_id,
		);
	}
}
