<?php

namespace Uncanny_Automator_Pro;

use Exception;

/**
 * Class EDD_USER_DOWNLOADED_FILE
 *
 * @package Uncanny_Automator_Pro
 */
class EDD_USER_DOWNLOADED_FILE extends \Uncanny_Automator_Pro\Action_Condition {

	/**
	 * Defines the condition.
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration   = 'EDD';
		$this->name          = esc_html_x( 'The user {{has/has not}} downloaded {{a file}}', 'Easy Digital Downloads', 'uncanny-automator-pro' );
		$this->code          = 'EDD_DOWNLOADED_FILE';
		$this->requires_user = true;
		$this->dynamic_name  = sprintf(
			// translators: %1$s: Condition, %2$s: Download
			esc_html_x( 'The user {{%1$s}} downloaded {{a file:%2$s}}', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'CONDITION',
			'EDD_DOWNLOAD'
		);
	}

	/**
	 * Defines the fields.
	 *
	 * @return array
	 */
	public function fields() {

		$edd_helper = new \Uncanny_Automator_Pro\Integrations\Easy_Digital_Downloads\Edd_Pro_Helpers();

		return array(
			array(
				'option_code' => 'CONDITION',
				'label'       => esc_html_x( 'Condition', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'input_type'  => 'select',
				'required'    => true,
				'options'     => array(
					array(
						'text'  => esc_html_x( 'has', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
						'value' => 'has',
					),
					array(
						'text'  => esc_html_x( 'has not', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
						'value' => 'has-not',
					),
				),
			),
			$edd_helper->all_edd_downloads( esc_html_x( 'Download', 'Easy Digital Downloads', 'uncanny-automator' ), 'EDD_DOWNLOAD', false, false, false ),
		);
	}

	/**
	 * Check if EDD integration dependencies are active.
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return function_exists( 'edd_has_user_purchased' ) && function_exists( 'edd_count_file_download_logs' );
	}

	/**
	 * Evaluates the condition.
	 *
	 * Has to use the $this->condition_failed( $message ); method if the condition is not met.
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$download_id = absint( $this->get_parsed_option( 'EDD_DOWNLOAD' ) );
		$condition   = trim( $this->get_parsed_option( 'CONDITION' ) );
		// @todo: Introduce a getter method in the parent class so the IDE does not complain the that property $user_id is not defined.
		$user_id = absint( $this->user_id );

		if ( ! function_exists( 'edd_has_user_purchased' ) ) {
			$this->condition_failed( 'The function "edd_has_user_purchased" is not defined' );

			return;
		}

		//      if ( false === edd_has_user_purchased( $user_id, $download_id ) ) {
		//          $this->condition_failed( sprintf( 'The user (ID: %1$d) has not purchased the specified download (ID: %2$d)', $user_id, $download_id ) );
		//
		//          return;
		//      }

		if ( ! function_exists( 'edd_count_file_download_logs' ) ) {
			$this->condition_failed( 'The function "edd_count_file_download_logs" is not defined' );

			return;
		}

		$customer = edd_get_customer_by( 'user_id', $user_id );
		if ( ! $customer ) {
			$this->condition_failed( sprintf( 'The user (ID: %1$d) is not a customer', $user_id ) );

			return;
		}

		$customer_id = $customer->id;

		$download_count = edd_count_file_download_logs(
			array(
				'product_id'  => $download_id,
				'customer_id' => $customer_id,
			)
		);

		if ( 'has-not' === $condition && $download_count > 0 ) {
			$this->condition_failed( sprintf( 'The user (ID: %1$d) has downloaded the specified download (ID: %2$d)', $user_id, $download_id ) );

			return;
		}

		if ( 'has' === $condition && $download_count <= 0 ) {
			$this->condition_failed( sprintf( 'The user (ID: %1$d) has not downloaded the specified download (ID: %2$d)', $user_id, $download_id ) );

			return;
		}
	}
}
