<?php

namespace Uncanny_Automator_Pro\Integrations\Easy_Digital_Downloads;

/**
 * Class EDD_GENERATE_DISCOUNT_CODE
 *
 * @package Uncanny_Automator_Pro
 */
class EDD_GENERATE_DISCOUNT_CODE extends \Uncanny_Automator\Recipe\Action {

	/**
	 * Define and register the action.
	 */
	public function setup_action() {
		$this->set_integration( 'EDD' );
		$this->set_action_code( 'EDD_GENERATE_DISCOUNT_CODE' );
		$this->set_action_meta( 'EDD_DISCOUNT_NAME' );
		$this->set_is_pro( true );
		$this->set_requires_user( false );
		$this->set_sentence(
			sprintf(
				// translators: %1$s: discount code
				esc_html_x( 'Generate a {{discount code:%1$s}}', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				$this->get_action_meta()
			)
		);
		$this->set_readable_sentence( esc_html_x( 'Generate a discount code', 'Easy Digital Downloads', 'uncanny-automator-pro' ) );
	}

	/**
	 * Define the Action's options
	 *
	 * @return array
	 */
	public function options() {
		// Required fields
		$discount_name = array(
			'input_type'      => 'text',
			'option_code'     => $this->get_action_meta(),
			'label'           => esc_html_x( 'Discount code name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'        => true,
			'description'     => esc_html_x( 'Enter a name for this discount code', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'relevant_tokens' => array(),
		);

		$discount_code = array(
			'input_type'      => 'text',
			'option_code'     => $this->get_action_meta() . '_CODE',
			'label'           => esc_html_x( 'Discount code', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'        => false, //optional: we will generate a random code if this is empty
			'description'     => esc_html_x( 'Enter the code customers will use (leave empty to auto-generate)', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'relevant_tokens' => array(),
		);

		$discount_amount = array(
			'input_type'      => 'int',
			'option_code'     => $this->get_action_meta() . '_AMOUNT',
			'label'           => esc_html_x( 'Discount amount', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'        => true,
			'relevant_tokens' => array(),
		);

		$discount_type = array(
			'input_type'            => 'select',
			'option_code'           => $this->get_action_meta() . '_TYPE',
			'label'                 => esc_html_x( 'Discount type', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'              => true,
			'options'               => $this->get_discount_type_options(),
			'options_show_id'       => false,
			'default_value'         => 'percent',
			'supports_custom_value' => false,
			'relevant_tokens'       => array(),
		);

		// Optional fields
		$discount_status = array(
			'input_type'            => 'select',
			'option_code'           => $this->get_action_meta() . '_STATUS',
			'label'                 => esc_html_x( 'Status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'              => false,
			'options'               => $this->get_discount_status_options(),
			'options_show_id'       => false,
			'default_value'         => 'active',
			'supports_custom_value' => false,
			'relevant_tokens'       => array(),
		);

		$start_date = array(
			'input_type'      => 'date',
			'option_code'     => $this->get_action_meta() . '_START_DATE',
			'label'           => esc_html_x( 'Start date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'relevant_tokens' => array(),
		);

		$end_date = array(
			'input_type'      => 'date',
			'option_code'     => $this->get_action_meta() . '_END_DATE',
			'label'           => esc_html_x( 'End date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'relevant_tokens' => array(),
		);

		$max_uses = array(
			'input_type'      => 'int',
			'option_code'     => $this->get_action_meta() . '_MAX_USES',
			'label'           => esc_html_x( 'Maximum uses', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'description'     => esc_html_x( 'Maximum times this discount can be used (0 = unlimited)', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'relevant_tokens' => array(),
		);

		$min_amount = array(
			'input_type'      => 'float',
			'option_code'     => $this->get_action_meta() . '_MIN_AMOUNT',
			'label'           => esc_html_x( 'Minimum purchase amount', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'relevant_tokens' => array(),
		);

		$once_per_customer = array(
			'input_type'            => 'select',
			'option_code'           => $this->get_action_meta() . '_ONCE_PER_CUSTOMER',
			'label'                 => esc_html_x( 'Once Per Customer', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'              => false,
			'options'               => $this->get_once_per_customer_options(),
			'default_value'         => 'no',
			'options_show_id'       => false,
			'supports_custom_value' => false,
			'description'           => esc_html_x( 'Limit to one use per customer', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'relevant_tokens'       => array(),
		);

		// Product Restrictions (Optional)
		$product_condition = array(
			'input_type'            => 'select',
			'option_code'           => $this->get_action_meta() . '_PRODUCT_CONDITION',
			'label'                 => esc_html_x( 'Product Condition', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'              => false,
			'options'               => $this->get_product_condition_options(),
			'options_show_id'       => false,
			'default_value'         => 'any',
			'supports_custom_value' => false,
			'relevant_tokens'       => array(),
		);

		$applicable_products = $this->get_item_helpers()->all_edd_downloads_not_required( esc_html_x( 'Select products required for this discount', 'Easy Digital Downloads', 'uncanny-automator-pro' ), $this->get_action_meta() . '_PRODUCTS', true, false );

		$excluded_products = $this->get_item_helpers()->all_edd_downloads_not_required( esc_html_x( 'Select products excluded from this discount', 'Easy Digital Downloads', 'uncanny-automator-pro' ), $this->get_action_meta() . '_EXCLUDED_PRODUCTS', true, false );

		$scope = array(
			'input_type'            => 'select',
			'option_code'           => $this->get_action_meta() . '_SCOPE',
			'label'                 => esc_html_x( 'Scope', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'required'              => false,
			'options'               => $this->get_scope_options(),
			'options_show_id'       => false,
			'default_value'         => 'global',
			'supports_custom_value' => false,
			'relevant_tokens'       => array(),
		);

		return array(
			$discount_name,
			$discount_code,
			$discount_amount,
			$discount_type,
			$discount_status,
			$start_date,
			$end_date,
			$max_uses,
			$min_amount,
			$once_per_customer,
			$product_condition,
			$applicable_products,
			$excluded_products,
			$scope,
		);
	}


	/**
	 * Process the action.
	 *
	 * @param int   $user_id
	 * @param array $action_data
	 * @param int   $recipe_id
	 * @param array $args
	 * @param array $parsed
	 *
	 * @return bool
	 */
	public function process_action( $user_id, $action_data, $recipe_id, $args, $parsed ) {
		// Check if EDD is active
		if ( ! function_exists( 'edd_add_discount' ) ) {
			$this->add_log_error( 'EDD is not active or edd_add_discount function not available.' );
			return false;
		}

		// Get required fields
		$name        = sanitize_text_field( $parsed[ $this->get_action_meta() ] );
		$code        = sanitize_text_field( $parsed[ $this->get_action_meta() . '_CODE' ] );
		$amount      = floatval( $parsed[ $this->get_action_meta() . '_AMOUNT' ] );
		$amount_type = sanitize_text_field( $parsed[ $this->get_action_meta() . '_TYPE' ] );

		// Validate required fields
		if ( empty( $name ) ) {
			$this->add_log_error( 'Discount name is required.' );
			return false;
		}

		if ( empty( $amount ) ) {
			$this->add_log_error( 'Discount amount is required.' );
			return false;
		}

		if ( empty( $amount_type ) ) {
			$this->add_log_error( 'Discount amount type is required.' );
			return false;
		}

		// Validate amount type
		if ( ! in_array( $amount_type, array( 'percent', 'flat' ), true ) ) {
			$this->add_log_error( 'Amount type must be either "percent" or "flat".' );
			return false;
		}

		// Validate amount values
		if ( 'percent' === $amount_type && ( $amount < 0 || $amount > 100 ) ) {
			$this->add_log_error( 'Percentage amount must be between 0 and 100.' );
			return false;
		}

		if ( 'flat' === $amount_type && $amount < 0 ) {
			$this->add_log_error( 'Flat amount cannot be negative.' );
			return false;
		}

		// Generate code if not provided
		if ( empty( $code ) ) {
			$code = $this->generate_unique_discount_code();
			if ( is_wp_error( $code ) ) {
				$this->add_log_error( 'Failed to generate unique discount code: ' . $code->get_error_message() );
				return false;
			}
		}

		// Check if code already exists
		if ( $this->discount_code_exists( $code ) ) {
			$this->add_log_error( sprintf( 'Discount code "%s" already exists.', $code ) );
			return false;
		}

		// Prepare discount data with defaults
		$discount_data = array(
			'name'        => $name,
			'code'        => $code,
			'amount'      => $amount,
			'amount_type' => $amount_type,
			'status'      => 'active', // Default status
		);

		// Add optional fields
		if ( ! empty( $parsed[ $this->get_action_meta() . '_STATUS' ] ) ) {
			$discount_data['status'] = $parsed[ $this->get_action_meta() . '_STATUS' ];
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_START_DATE' ] ) ) {
			$discount_data['start_date'] = $parsed[ $this->get_action_meta() . '_START_DATE' ] . ' 00:00:00';
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_END_DATE' ] ) ) {
			$discount_data['end_date'] = $parsed[ $this->get_action_meta() . '_END_DATE' ] . ' 23:59:59';
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_MAX_USES' ] ) && is_numeric( $parsed[ $this->get_action_meta() . '_MAX_USES' ] ) ) {
			$discount_data['max_uses'] = absint( $parsed[ $this->get_action_meta() . '_MAX_USES' ] );
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_MIN_AMOUNT' ] ) && is_numeric( $parsed[ $this->get_action_meta() . '_MIN_AMOUNT' ] ) ) {
			$discount_data['min_charge_amount'] = floatval( $parsed[ $this->get_action_meta() . '_MIN_AMOUNT' ] );
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_ONCE_PER_CUSTOMER' ] ) && 'yes' === $parsed[ $this->get_action_meta() . '_ONCE_PER_CUSTOMER' ] ) {
			$discount_data['once_per_customer'] = true;
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_PRODUCT_CONDITION' ] ) ) {
			$discount_data['product_condition'] = $parsed[ $this->get_action_meta() . '_PRODUCT_CONDITION' ];
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_PRODUCTS' ] ) && '-1' !== $parsed[ $this->get_action_meta() . '_PRODUCTS' ] ) {
			$discount_data['product_reqs'] = array( absint( $parsed[ $this->get_action_meta() . '_PRODUCTS' ] ) );
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_EXCLUDED_PRODUCTS' ] ) && '-1' !== $parsed[ $this->get_action_meta() . '_EXCLUDED_PRODUCTS' ] ) {
			$discount_data['excluded_products'] = array( absint( $parsed[ $this->get_action_meta() . '_EXCLUDED_PRODUCTS' ] ) );
		}

		if ( ! empty( $parsed[ $this->get_action_meta() . '_SCOPE' ] ) ) {
			$discount_data['scope'] = $parsed[ $this->get_action_meta() . '_SCOPE' ];
		}

		// Validate dates if both are set
		if ( ! empty( $discount_data['start_date'] ) && ! empty( $discount_data['end_date'] ) ) {
			$start_timestamp = strtotime( $discount_data['start_date'] );
			$end_timestamp   = strtotime( $discount_data['end_date'] );

			if ( $start_timestamp > $end_timestamp ) {
				$this->add_log_error( 'Start date cannot be after end date.' );
				return false;
			}

			// Check if end date has passed
			if ( $end_timestamp < current_time( 'timestamp' ) ) { //phpcs:ignore WordPress.DateTime.CurrentTimeTimestamp.Requested
				$discount_data['status'] = 'expired';
			}
		}

		// Validate product condition
		if ( ! empty( $discount_data['product_condition'] ) && ! in_array( $discount_data['product_condition'], array( 'any', 'all' ), true ) ) {
			$this->add_log_error( 'Product condition must be either "any" or "all".' );
			return false;
		}

		// Add action hooks
		do_action( 'edd_pre_insert_discount', $discount_data );

		// Create the discount
		$discount_id = edd_add_discount( $discount_data );

		// Check for errors
		if ( is_wp_error( $discount_id ) ) {
			$this->add_log_error( sprintf( 'Failed to create discount code: %s', $discount_id->get_error_message() ) );
			return false;
		}

		if ( ! $discount_id ) {
			$this->add_log_error( 'Failed to create discount code. Unknown error occurred.' );
			return false;
		}

		// Success - add action hooks
		do_action( 'edd_post_insert_discount', $discount_data, $discount_id );

		// Hydrate tokens so the generated code can be used elsewhere
		$this->hydrate_tokens(
			array(
				'DISCOUNT_CODE'       => $code,
				'DISCOUNT_NAME'       => $name,
				'DISCOUNT_AMOUNT'     => $amount,
				'DISCOUNT_TYPE'       => $amount_type,
				'DISCOUNT_ID'         => $discount_id,
				'DISCOUNT_STATUS'     => $this->format_status_field( $parsed[ $this->get_action_meta() . '_STATUS' ] ?? '' ),
				'START_DATE'          => $this->format_date_field( $parsed[ $this->get_action_meta() . '_START_DATE' ] ?? '' ),
				'END_DATE'            => $this->format_date_field( $parsed[ $this->get_action_meta() . '_END_DATE' ] ?? '' ),
				'MAX_USES'            => $this->format_empty_field( $parsed[ $this->get_action_meta() . '_MAX_USES' ] ?? '', 'unlimited' ),
				'MIN_AMOUNT'          => $this->format_empty_field( $parsed[ $this->get_action_meta() . '_MIN_AMOUNT' ] ?? '', 'no minimum' ),
				'ONCE_PER_CUSTOMER'   => $this->format_boolean_field( $parsed[ $this->get_action_meta() . '_ONCE_PER_CUSTOMER' ] ?? '' ),
				'PRODUCT_CONDITION'   => $this->format_product_condition_field( $parsed[ $this->get_action_meta() . '_PRODUCT_CONDITION' ] ?? '' ),
				'APPLICABLE_PRODUCTS' => $this->format_product_field( $parsed[ $this->get_action_meta() . '_PRODUCTS' ] ?? '' ),
				'EXCLUDED_PRODUCTS'   => $this->format_product_field( $parsed[ $this->get_action_meta() . '_EXCLUDED_PRODUCTS' ] ?? '' ),
				'SCOPE'               => $this->format_scope_field( $parsed[ $this->get_action_meta() . '_SCOPE' ] ?? '' ),
			)
		);

		return true;
	}

	/**
	 * Generate a unique discount code.
	 *
	 * @param int    $length        Length of the code
	 * @param string $prefix        Prefix for the code
	 * @param bool   $include_letters Whether to include letters
	 * @param bool   $include_numbers Whether to include numbers
	 * @param int    $max_attempts  Maximum attempts to generate unique code
	 *
	 * @return string|WP_Error Unique discount code or WP_Error
	 */
	private function generate_unique_discount_code( $length = 8, $prefix = '', $include_letters = true, $include_numbers = true, $max_attempts = 100 ) {
		if ( ! $include_letters && ! $include_numbers ) {
			return new WP_Error( 'invalid_chars', 'Must include either letters or numbers.' );
		}

		$attempts = 0;

		do {
			$code = $this->generate_random_discount_code( $length, $prefix, $include_letters, $include_numbers );

			if ( is_wp_error( $code ) ) {
				return $code;
			}

			// Check if code already exists
			if ( ! $this->discount_code_exists( $code ) ) {
				return $code;
			}

			++$attempts;

		} while ( $attempts < $max_attempts );

		return new WP_Error( 'max_attempts_reached', 'Unable to generate unique discount code after maximum attempts.' );
	}

	/**
	 * Generate a random discount code.
	 *
	 * @param int    $length        Length of the code
	 * @param string $prefix        Prefix for the code
	 * @param bool   $include_letters Whether to include letters
	 * @param bool   $include_numbers Whether to include numbers
	 *
	 * @return string|WP_Error Generated discount code or WP_Error
	 */
	private function generate_random_discount_code( $length = 8, $prefix = '', $include_letters = true, $include_numbers = true ) {
		if ( ! $include_letters && ! $include_numbers ) {
			return new WP_Error( 'invalid_chars', 'Must include either letters or numbers.' );
		}

		$characters = '';

		if ( $include_letters ) {
			$characters .= 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
		}

		if ( $include_numbers ) {
			$characters .= '0123456789';
		}

		$code = '';

		for ( $i = 0; $i < $length; $i++ ) {
			$code .= $characters[ wp_rand( 0, strlen( $characters ) - 1 ) ];
		}

		// Add prefix if provided
		if ( ! empty( $prefix ) ) {
			$code = $prefix . $code;
		}

		return $code;
	}

	/**
	 * Check if a discount code already exists.
	 *
	 * @param string $code
	 *
	 * @return bool
	 */
	private function discount_code_exists( $code ) {
		global $wpdb;

		$exists = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->prefix}edd_discounts WHERE code = %s",
				$code
			)
		);

		return (bool) $exists;
	}

	/**
	 * Define tokens for this action.
	 *
	 * @return array
	 */
	public function define_tokens() {
		return array(
			'DISCOUNT_CODE'       => array(
				'name' => esc_html_x( 'Generated discount code', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'DISCOUNT_NAME'       => array(
				'name' => esc_html_x( 'Discount name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'DISCOUNT_AMOUNT'     => array(
				'name' => esc_html_x( 'Discount amount', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'int',
			),
			'DISCOUNT_TYPE'       => array(
				'name' => esc_html_x( 'Discount type', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'DISCOUNT_ID'         => array(
				'name' => esc_html_x( 'Discount ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'int',
			),
			'DISCOUNT_STATUS'     => array(
				'name' => esc_html_x( 'Discount status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'START_DATE'          => array(
				'name' => esc_html_x( 'Start date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'END_DATE'            => array(
				'name' => esc_html_x( 'End date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'MAX_USES'            => array(
				'name' => esc_html_x( 'Maximum uses', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'int',
			),
			'MIN_AMOUNT'          => array(
				'name' => esc_html_x( 'Minimum purchase amount', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'float',
			),
			'ONCE_PER_CUSTOMER'   => array(
				'name' => esc_html_x( 'Once per customer', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'PRODUCT_CONDITION'   => array(
				'name' => esc_html_x( 'Product condition', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'APPLICABLE_PRODUCTS' => array(
				'name' => esc_html_x( 'Applicable products', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'EXCLUDED_PRODUCTS'   => array(
				'name' => esc_html_x( 'Excluded products', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'SCOPE'               => array(
				'name' => esc_html_x( 'Scope', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
		);
	}

	/**
	 * Get discount type options.
	 *
	 * @return mixed
	 */
	private function get_discount_type_options() {
		return array(
			array(
				'text'  => esc_html_x( 'Percentage (%)', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'percent',
			),
			array(
				'text'  => esc_html_x( 'Fixed Amount ($)', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'flat',
			),
		);
	}
	/**
	 * Get discount status options.
	 *
	 * @return mixed
	 */
	private function get_discount_status_options() {
		return array(
			array(
				'text'  => esc_html_x( 'Active', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'active',
			),
			array(
				'text'  => esc_html_x( 'Inactive', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'inactive',
			),
			array(
				'text'  => esc_html_x( 'Expired', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'expired',
			),
			array(
				'text'  => esc_html_x( 'Archived', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'archived',
			),
		);
	}
	/**
	 * Get product condition options.
	 *
	 * @return mixed
	 */
	private function get_product_condition_options() {
		return array(
			array(
				'text'  => esc_html_x( 'Any of the selected products', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'any',
			),
			array(
				'text'  => esc_html_x( 'All of the selected products', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'all',
			),
		);
	}
	/**
	 * Get scope options.
	 *
	 * @return mixed
	 */
	private function get_scope_options() {
		return array(
			array(
				'text'  => esc_html_x( 'Global (All Products)', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'global',
			),
			array(
				'text'  => esc_html_x( 'Product Specific', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'product_specific',
			),
			array(
				'text'  => esc_html_x( 'Category Specific', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'category_specific',
			),
		);
	}
	/**
	 * Get once per customer options.
	 *
	 * @return mixed
	 */
	private function get_once_per_customer_options() {
		return array(
			array(
				'text'  => esc_html_x( 'No', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'no',
			),
			array(
				'text'  => esc_html_x( 'Yes', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'value' => 'yes',
			),
		);
	}

	/**
	 * Format product field values for tokens.
	 * Converts -1 to 'any' for better readability.
	 *
	 * @param mixed $value The field value.
	 * @return string The formatted value.
	 */
	private function format_product_field( $value ) {
		if ( '-1' === $value || -1 === $value ) {
			return 'any';
		}

		if ( empty( $value ) ) {
			return '';
		}

		return $value;
	}

	/**
	 * Format status field values for tokens.
	 * Capitalizes first letter for better readability.
	 *
	 * @param string $value The status value.
	 * @return string The formatted status.
	 */
	private function format_status_field( $value ) {
		if ( empty( $value ) ) {
			return '';
		}

		return ucfirst( $value );
	}

	/**
	 * Format date field values for tokens.
	 * Shows friendly text for empty dates.
	 *
	 * @param string $value The date value.
	 * @return string The formatted date or friendly text.
	 */
	private function format_date_field( $value ) {
		if ( empty( $value ) ) {
			return 'not set';
		}

		return $value;
	}

	/**
	 * Format empty field values for tokens.
	 * Shows friendly text for empty values.
	 *
	 * @param mixed $value The field value.
	 * @param string $friendly_text The text to show when empty.
	 * @return string The formatted value or friendly text.
	 */
	private function format_empty_field( $value, $friendly_text ) {
		if ( empty( $value ) ) {
			return $friendly_text;
		}

		return $value;
	}

	/**
	 * Format boolean field values for tokens.
	 * Capitalizes first letter for better readability.
	 *
	 * @param string $value The boolean value.
	 * @return string The formatted boolean value.
	 */
	private function format_boolean_field( $value ) {
		if ( empty( $value ) ) {
			return '';
		}

		return ucfirst( $value );
	}

	/**
	 * Format product condition field values for tokens.
	 * Shows more descriptive text.
	 *
	 * @param string $value The product condition value.
	 * @return string The formatted product condition.
	 */
	private function format_product_condition_field( $value ) {
		if ( empty( $value ) ) {
			return '';
		}

		switch ( $value ) {
			case 'any':
				return 'Any of selected';
			case 'all':
				return 'All of selected';
			default:
				return ucfirst( $value );
		}
	}

	/**
	 * Format scope field values for tokens.
	 * Shows more descriptive text.
	 *
	 * @param string $value The scope value.
	 * @return string The formatted scope.
	 */
	private function format_scope_field( $value ) {
		if ( empty( $value ) ) {
			return '';
		}

		switch ( $value ) {
			case 'global':
				return 'Global (All Products)';
			case 'product_specific':
				return 'Product Specific';
			case 'category_specific':
				return 'Category Specific';
			default:
				return ucfirst( $value );
		}
	}
}
