<?php

namespace Uncanny_Automator_Pro\Integrations\EDD_SL;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class EDD_USER_LICENSE_DISABLED
 *
 * @package Uncanny_Automator_Pro
 * @method Uncanny_Automator\Integrations\Edd_SL\Edd_Sl_Helpers get_item_helpers()
 */
class EDD_USER_LICENSE_DISABLED extends Trigger {

	/**
	 * Trigger code.
	 *
	 * @var string
	 */
	protected $trigger_code = 'EDD_USER_LICENSE_DISABLED';

	/**
	 * Trigger meta.
	 *
	 * @var string
	 */
	protected $trigger_meta = 'EDDPRODUCT';


	/**
	 * Set up Automator trigger.
	 */
	protected function setup_trigger() {
		$this->set_integration( 'EDD_SL' );
		$this->set_trigger_code( $this->trigger_code );
		$this->set_trigger_meta( $this->trigger_meta );
		$this->set_is_pro( true );
		$this->set_trigger_type( 'user' );
		$this->set_is_login_required( false );
		$this->set_support_link( Automator()->get_author_support_link( $this->trigger_code, 'integration/easy-digital-downloads/' ) );
		// translators: %1$s: download
		$this->set_sentence( sprintf( esc_html_x( "A user's license for {{a download:%1\$s}} is disabled", 'Easy Digital Downloads', 'uncanny-automator-pro' ), $this->get_trigger_meta() ) );
		$this->set_readable_sentence( esc_html_x( "A user's license for {{a download}} is disabled", 'Easy Digital Downloads', 'uncanny-automator-pro' ) );
		$this->add_action( 'edd_sl_post_set_status', 10, 2 );
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			Automator()->helpers->recipe->field->select(
				array(
					'option_code'     => $this->get_trigger_meta(),
					'label'           => esc_html_x( 'Download', 'EDD - Software Licensing', 'uncanny-automator' ),
					'options'         => $this->get_item_helpers()->get_all_downloads(),
					'relevant_tokens' => array(),
				)
			),
		);
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger
	 * @param array $tokens
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		return array(
			'EDD_DOWNLOAD_TITLE'           => array(
				'name'      => esc_html_x( 'Download title', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_DOWNLOAD_TITLE',
				'tokenName' => esc_html_x( 'Download title', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_ID'              => array(
				'name'      => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_ID',
				'tokenName' => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_URL'             => array(
				'name'      => esc_html_x( 'Download URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_URL',
				'tokenName' => esc_html_x( 'Download URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_LICENSE_KEY'              => array(
				'name'      => esc_html_x( 'License key', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_LICENSE_KEY',
				'tokenName' => esc_html_x( 'License key', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_LICENSE_ID'               => array(
				'name'      => esc_html_x( 'License ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_LICENSE_ID',
				'tokenName' => esc_html_x( 'License ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_LICENSE_STATUS'           => array(
				'name'      => esc_html_x( 'License status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_LICENSE_STATUS',
				'tokenName' => esc_html_x( 'License status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_LICENSE_EXPIRATION'       => array(
				'name'      => esc_html_x( 'License expiration date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'date',
				'tokenId'   => 'EDD_LICENSE_EXPIRATION',
				'tokenName' => esc_html_x( 'License expiration date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_LICENSE_ACTIVATION_LIMIT' => array(
				'name'      => esc_html_x( 'License activation limit', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_LICENSE_ACTIVATION_LIMIT',
				'tokenName' => esc_html_x( 'License activation limit', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_LICENSE_ACTIVATION_COUNT' => array(
				'name'      => esc_html_x( 'License activation count', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_LICENSE_ACTIVATION_COUNT',
				'tokenName' => esc_html_x( 'License activation count', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_ID'              => array(
				'name'      => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_ID',
				'tokenName' => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_EMAIL'           => array(
				'name'      => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_EMAIL',
				'tokenName' => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_NAME'            => array(
				'name'      => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_NAME',
				'tokenName' => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_STATUS'          => array(
				'name'      => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_STATUS',
				'tokenName' => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_USER_ID'         => array(
				'name'      => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_USER_ID',
				'tokenName' => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( empty( $hook_args ) || count( $hook_args ) < 2 ) {
			return false;
		}

		if ( ! isset( $trigger['meta'][ $this->trigger_meta ] ) ) {
			return false;
		}

		$selected_download = $trigger['meta'][ $this->trigger_meta ];

		// edd_sl_post_set_status parameters: $license_id, $status
		$license_id = $hook_args[0];
		$status     = $hook_args[1];

		// Only trigger when license is disabled
		if ( 'disabled' !== $status ) {
			return false;
		}

		// Get license object
		$license = new \EDD_SL_License( $license_id );
		if ( ! $license || empty( $license->ID ) ) {
			return false;
		}

		$download_id = $license->download_id;

		// Set user ID for the trigger from customer data
		$user_id = 0;
		if ( ! empty( $license->customer_id ) ) {
			$customer = \edd_get_customer( $license->customer_id );
			if ( $customer && ! empty( $customer->user_id ) ) {
				$user_id = $customer->user_id;
			}
		}

		// Fallback to current user ID if no customer user ID found
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( $user_id ) {
			$this->set_user_id( $user_id );
		}

		// Check if download matches
		if ( intval( '-1' ) !== intval( $selected_download ) && absint( $selected_download ) !== absint( $download_id ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		// edd_sl_post_set_status parameters: $license_id, $status
		$license_id = $hook_args[0];
		$status     = $hook_args[1];

		// Get license object
		$license = new \EDD_SL_License( $license_id );
		if ( ! $license || empty( $license->ID ) ) {
			return array();
		}

		// Get customer information
		$customer_id      = '';
		$customer_email   = '';
		$customer_name    = '';
		$customer_status  = '';
		$customer_user_id = '';

		if ( ! empty( $license->customer_id ) ) {
			$customer = \edd_get_customer( $license->customer_id );
			if ( $customer ) {
				$customer_id      = $customer->id;
				$customer_email   = $customer->email;
				$customer_name    = $customer->name;
				$customer_status  = $customer->status;
				$customer_user_id = $customer->user_id;
			}
		}

		$token_values = array(
			'EDD_DOWNLOAD_TITLE'           => get_the_title( $license->download_id ),
			'EDD_DOWNLOAD_ID'              => $license->download_id,
			'EDD_DOWNLOAD_URL'             => get_permalink( $license->download_id ),
			'EDD_LICENSE_KEY'              => $license->license_key,
			'EDD_LICENSE_ID'               => $license->ID,
			'EDD_LICENSE_STATUS'           => $license->status,
			'EDD_LICENSE_EXPIRATION'       => $license->expiration ? date_i18n( get_option( 'date_format' ), strtotime( $license->expiration ) ) : '',
			'EDD_LICENSE_ACTIVATION_LIMIT' => $license->activation_limit,
			'EDD_LICENSE_ACTIVATION_COUNT' => $license->activation_count,
			'EDD_CUSTOMER_ID'              => $customer_id,
			'EDD_CUSTOMER_EMAIL'           => $customer_email,
			'EDD_CUSTOMER_NAME'            => $customer_name,
			'EDD_CUSTOMER_STATUS'          => $customer_status,
			'EDD_CUSTOMER_USER_ID'         => $customer_user_id,
		);

		return $token_values;
	}
}
