<?php

namespace Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class EDD_USER_SUBSCRIPTION_RENEWED_PRODWITHPRICEOPTION
 *
 * @package Uncanny_Automator_Pro
 */
class EDD_USER_SUBSCRIPTION_RENEWED_PRODWITHPRICEOPTION extends Trigger {

	/**
	 * Trigger code.
	 *
	 * @var string
	 */
	protected $trigger_code = 'EDD_USER_SUBSCRIPTION_RENEWED_PRODWITHPRICEOPTION';

	/**
	 * Trigger meta.
	 *
	 * @var string
	 */
	protected $trigger_meta = 'EDDPRODUCT';

	/**
	 * Price option meta.
	 *
	 * @var string
	 */
	protected $price_option_meta = 'EDDPRICEOPTION';


	/**
	 * Set up Automator trigger.
	 */
	protected function setup_trigger() {
		$this->set_integration( 'EDD_RECURRING' );
		$this->set_trigger_code( $this->trigger_code );
		$this->set_trigger_meta( $this->trigger_meta );
		$this->set_is_pro( true );
		$this->set_trigger_type( 'user' );
		$this->set_is_login_required( false );
		$this->set_support_link( Automator()->get_author_support_link( $this->trigger_code, 'integration/easy-digital-downloads/' ) );
		// translators: %1$s: download, %2$s: price option
		$this->set_sentence( sprintf( esc_html_x( "A user's subscription to {{a price option:%2\$s}} of {{a download:%1\$s}} is renewed", 'Easy Digital Downloads', 'uncanny-automator-pro' ), $this->get_trigger_meta(), $this->price_option_meta . ':' . $this->get_trigger_meta() ) );
		$this->set_readable_sentence( esc_html_x( "A user's subscription to {{a price option}} of {{a download}} is renewed", 'Easy Digital Downloads', 'uncanny-automator-pro' ) );
		$this->add_action( 'edd_subscription_post_renew', 10, 4 );
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			$this->get_item_helpers()->all_edd_downloads( esc_html_x( 'Download', 'Easy Digital Downloads', 'uncanny-automator' ), $this->get_trigger_meta(), true, false ),
			array(
				'option_code'     => $this->price_option_meta,
				'label'           => esc_html_x( 'Price option', 'Easy Digital Downloads', 'uncanny-automator' ),
				'input_type'      => 'select',
				'required'        => true,
				'options'         => array(),
				'relevant_tokens' => array(),
				'ajax'            => array(
					'event'         => 'parent_fields_change',
					'endpoint'      => 'automator_edd_price_options_handler',
					'listen_fields' => array( $this->get_trigger_meta() ),
				),
			),
		);
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger
	 * @param array $tokens
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		return array(
			'EDD_DOWNLOAD_TITLE'              => array(
				'name'      => esc_html_x( 'Download title', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_DOWNLOAD_TITLE',
				'tokenName' => esc_html_x( 'Download title', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_ID'                 => array(
				'name'      => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_ID',
				'tokenName' => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_URL'                => array(
				'name'      => esc_html_x( 'Download URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_URL',
				'tokenName' => esc_html_x( 'Download URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => array(
				'name'      => esc_html_x( 'Download featured image ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_ID',
				'tokenName' => esc_html_x( 'Download featured image ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => array(
				'name'      => esc_html_x( 'Download featured image URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_URL',
				'tokenName' => esc_html_x( 'Download featured image URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_PRICE_OPTION_NAME'           => array(
				'name'      => esc_html_x( 'Price option name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PRICE_OPTION_NAME',
				'tokenName' => esc_html_x( 'Price option name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_PRICE_OPTION_ID'             => array(
				'name'      => esc_html_x( 'Price option ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_PRICE_OPTION_ID',
				'tokenName' => esc_html_x( 'Price option ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_SUBSCRIPTION_ID'             => array(
				'name'      => esc_html_x( 'Subscription ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_SUBSCRIPTION_ID',
				'tokenName' => esc_html_x( 'Subscription ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_SUBSCRIPTION_NEXT_PAYMENT'   => array(
				'name'      => esc_html_x( 'Next payment date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'date',
				'tokenId'   => 'EDD_SUBSCRIPTION_NEXT_PAYMENT',
				'tokenName' => esc_html_x( 'Next payment date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_PAYMENT_ID'                  => array(
				'name'      => esc_html_x( 'Payment ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_PAYMENT_ID',
				'tokenName' => esc_html_x( 'Payment ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_PAYMENT_AMOUNT'              => array(
				'name'      => esc_html_x( 'Payment amount', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PAYMENT_AMOUNT',
				'tokenName' => esc_html_x( 'Payment amount', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_ID'                 => array(
				'name'      => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_ID',
				'tokenName' => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_EMAIL'              => array(
				'name'      => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_EMAIL',
				'tokenName' => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_NAME'               => array(
				'name'      => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_NAME',
				'tokenName' => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_STATUS'             => array(
				'name'      => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_STATUS',
				'tokenName' => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_USER_ID'            => array(
				'name'      => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_USER_ID',
				'tokenName' => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( empty( $hook_args ) || count( $hook_args ) < 4 ) {
			return false;
		}

		if ( ! isset( $trigger['meta'][ $this->trigger_meta ] ) ) {
			return false;
		}

		$selected_download     = $trigger['meta'][ $this->trigger_meta ];
		$selected_price_option = $trigger['meta'][ $this->price_option_meta ];

		// edd_subscription_post_renew parameters: $subscription_id, $expiration, $subscription, $payment_id
		$subscription_id     = $hook_args[0];
		$expiration          = $hook_args[1];
		$subscription_object = $hook_args[2];
		$payment_id          = $hook_args[3];

		if ( ! $subscription_object ) {
			return false;
		}

		$download_id     = $subscription_object->product_id;
		$price_option_id = $subscription_object->price_id;

		// Set user ID for the trigger from customer data
		$user_id = 0;
		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = \edd_get_customer( $subscription_object->customer_id );
			if ( $customer && ! empty( $customer->user_id ) ) {
				$user_id = $customer->user_id;
			}
		}

		// Fallback to current user ID if no customer user ID found
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( $user_id ) {
			$this->set_user_id( $user_id );
		}

		// Check if download matches
		if ( intval( '-1' ) !== intval( $selected_download ) && absint( $selected_download ) !== absint( $download_id ) ) {
			return false;
		}

		// Check if price option matches
		if ( intval( '-1' ) !== intval( $selected_price_option ) && absint( $selected_price_option ) !== absint( $price_option_id ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		// edd_subscription_post_renew parameters: $subscription_id, $expiration, $subscription, $payment_id
		$subscription_id     = $hook_args[0];
		$expiration          = $hook_args[1];
		$subscription_object = $hook_args[2];
		$payment_id          = $hook_args[3];

		if ( ! $subscription_object ) {
			return array();
		}

		// Get payment object
		$payment = new \EDD_Payment( $payment_id );

		// Get customer information.
		$customer_id      = '';
		$customer_email   = '';
		$customer_name    = '';
		$customer_status  = '';
		$customer_user_id = '';

		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = \edd_get_customer( $subscription_object->customer_id );
			if ( $customer ) {
				$customer_id      = $customer->id;
				$customer_email   = $customer->email;
				$customer_name    = $customer->name;
				$customer_status  = $customer->status;
				$customer_user_id = $customer->user_id;
			}
		}

		// Get price option name
		$price_option_name = '';
		if ( $subscription_object->price_id && edd_has_variable_prices( $subscription_object->product_id ) ) {
			$prices            = edd_get_variable_prices( $subscription_object->product_id );
			$price_option_name = isset( $prices[ $subscription_object->price_id ]['name'] ) ? $prices[ $subscription_object->price_id ]['name'] : '';
		}

		$token_values = array(
			'EDD_DOWNLOAD_TITLE'              => get_the_title( $subscription_object->product_id ),
			'EDD_DOWNLOAD_ID'                 => $subscription_object->product_id,
			'EDD_DOWNLOAD_URL'                => get_permalink( $subscription_object->product_id ),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => get_post_thumbnail_id( $subscription_object->product_id ),
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => get_the_post_thumbnail_url( $subscription_object->product_id ),
			'EDD_PRICE_OPTION_NAME'           => $price_option_name,
			'EDD_PRICE_OPTION_ID'             => $subscription_object->price_id,
			'EDD_SUBSCRIPTION_ID'             => $subscription_object->id,
			'EDD_SUBSCRIPTION_NEXT_PAYMENT'   => date_i18n( get_option( 'date_format' ), strtotime( $expiration ) ),
			'EDD_PAYMENT_ID'                  => $payment_id,
			'EDD_PAYMENT_AMOUNT'              => edd_currency_filter( edd_format_amount( $payment->total ) ),
			'EDD_CUSTOMER_ID'                 => $customer_id,
			'EDD_CUSTOMER_EMAIL'              => $customer_email,
			'EDD_CUSTOMER_NAME'               => $customer_name,
			'EDD_CUSTOMER_STATUS'             => $customer_status,
			'EDD_CUSTOMER_USER_ID'            => $customer_user_id,
		);

		return $token_values;
	}
}
