<?php

namespace Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class EDD_USER_REFUND_PRODWITHPRICEOPTION
 *
 * @package Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration
 * @method \Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration\Edd_Recurring_Pro_Helpers get_item_helpers()
 */
class EDD_USER_REFUND_PRODWITHPRICEOPTION extends Trigger {

	/**
	 * Trigger code.
	 *
	 * @var string
	 */
	protected $trigger_code = 'EDD_USER_REFUND_PRODWITHPRICEOPTION';

	/**
	 * Trigger meta.
	 *
	 * @var string
	 */
	protected $trigger_meta = 'EDDPRODUCT';

	/**
	 * Price option meta.
	 *
	 * @var string
	 */
	protected $price_option_meta = 'EDDPRICEOPTION';

	/**
	 * Matched download ID from validation.
	 *
	 * @var int
	 */
	protected $matched_download_id = 0;

	/**
	 * Matched price option ID from validation.
	 *
	 * @var int|null
	 */
	protected $matched_price_option_id = null;


	/**
	 * Set up Automator trigger.
	 */
	protected function setup_trigger() {
		$this->set_integration( 'EDD_RECURRING' );
		$this->set_trigger_code( $this->trigger_code );
		$this->set_trigger_meta( $this->trigger_meta );
		$this->set_is_pro( true );
		$this->set_trigger_type( 'user' );
		$this->set_is_login_required( false );
		$this->set_support_link( Automator()->get_author_support_link( $this->trigger_code, 'integration/easy-digital-downloads/' ) );
		// translators: %1$s: download, %2$s: price option
		$this->set_sentence( sprintf( esc_html_x( 'A user receives a Stripe refund for their subscription to {{a price option:%2$s}} of {{a download:%1$s}}', 'EDD - Recurring Payments', 'uncanny-automator-pro' ), $this->get_trigger_meta(), $this->price_option_meta . ':' . $this->get_trigger_meta() ) );
		$this->set_readable_sentence( esc_html_x( 'A user receives a Stripe refund for their subscription to {{a price option}} of {{a download}}', 'EDD - Recurring Payments', 'uncanny-automator-pro' ) );
		$this->add_action( 'edds_payment_refunded', 10, 1 );
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			$this->get_item_helpers()->all_recurring_edd_downloads( esc_html_x( 'Download', 'EDD - Recurring Payments', 'uncanny-automator-pro' ), $this->get_trigger_meta() ),
			array(
				'option_code'     => $this->price_option_meta,
				'label'           => esc_html_x( 'Price option', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'input_type'      => 'select',
				'required'        => true,
				'options'         => array(),
				'relevant_tokens' => array(),
				'ajax'            => array(
					'event'         => 'parent_fields_change',
					'endpoint'      => 'automator_edd_price_options_handler',
					'listen_fields' => array( $this->get_trigger_meta() ),
				),
			),
		);
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger
	 * @param array $tokens
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		return array(
			'EDD_DOWNLOAD_TITLE'              => array(
				'name'      => esc_html_x( 'Download title', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_DOWNLOAD_TITLE',
				'tokenName' => esc_html_x( 'Download title', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_ID'                 => array(
				'name'      => esc_html_x( 'Download ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_ID',
				'tokenName' => esc_html_x( 'Download ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_URL'                => array(
				'name'      => esc_html_x( 'Download URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_URL',
				'tokenName' => esc_html_x( 'Download URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => array(
				'name'      => esc_html_x( 'Download featured image ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_ID',
				'tokenName' => esc_html_x( 'Download featured image ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => array(
				'name'      => esc_html_x( 'Download featured image URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_URL',
				'tokenName' => esc_html_x( 'Download featured image URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PRICE_OPTION_NAME'           => array(
				'name'      => esc_html_x( 'Price option name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PRICE_OPTION_NAME',
				'tokenName' => esc_html_x( 'Price option name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PRICE_OPTION_ID'             => array(
				'name'      => esc_html_x( 'Price option ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_PRICE_OPTION_ID',
				'tokenName' => esc_html_x( 'Price option ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_ORDER_ID'                    => array(
				'name'      => esc_html_x( 'Order ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_ORDER_ID',
				'tokenName' => esc_html_x( 'Order ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_ORDER_URL'                   => array(
				'name'      => esc_html_x( 'Order URL (Admin)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_ORDER_URL',
				'tokenName' => esc_html_x( 'Order URL (Admin)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_ORDER_URL_USER'              => array(
				'name'      => esc_html_x( 'Order URL (User)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_ORDER_URL_USER',
				'tokenName' => esc_html_x( 'Order URL (User)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_ORDER_DISCOUNTS'             => array(
				'name'      => esc_html_x( 'Order discounts', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_DISCOUNTS',
				'tokenName' => esc_html_x( 'Order discounts', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_ORDER_SUBTOTAL'              => array(
				'name'      => esc_html_x( 'Order subtotal', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_SUBTOTAL',
				'tokenName' => esc_html_x( 'Order subtotal', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_ORDER_TAX'                   => array(
				'name'      => esc_html_x( 'Order tax', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_TAX',
				'tokenName' => esc_html_x( 'Order tax', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_ORDER_TOTAL'                 => array(
				'name'      => esc_html_x( 'Order total', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_ORDER_TOTAL',
				'tokenName' => esc_html_x( 'Order total', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PAYMENT_METHOD'              => array(
				'name'      => esc_html_x( 'Payment method', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PAYMENT_METHOD',
				'tokenName' => esc_html_x( 'Payment method', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DISCOUNT_CODES'              => array(
				'name'      => esc_html_x( 'Discount codes used', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_DISCOUNT_CODES',
				'tokenName' => esc_html_x( 'Discount codes used', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_RECURRING_PERIOD'            => array(
				'name'      => esc_html_x( 'Recurring period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_RECURRING_PERIOD',
				'tokenName' => esc_html_x( 'Recurring period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_SIGNUP_FEE'                  => array(
				'name'      => esc_html_x( 'Signup fee', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_SIGNUP_FEE',
				'tokenName' => esc_html_x( 'Signup fee', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_TIMES'                       => array(
				'name'      => esc_html_x( 'Times', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_TIMES',
				'tokenName' => esc_html_x( 'Times', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_FREE_TRIAL_PERIOD'           => array(
				'name'      => esc_html_x( 'Free trial period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_FREE_TRIAL_PERIOD',
				'tokenName' => esc_html_x( 'Free trial period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_SUBSCRIPTION_ID'             => array(
				'name'      => esc_html_x( 'Subscription ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_SUBSCRIPTION_ID',
				'tokenName' => esc_html_x( 'Subscription ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_SUBSCRIPTION_URL'            => array(
				'name'      => esc_html_x( 'Subscription URL (Admin)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_SUBSCRIPTION_URL',
				'tokenName' => esc_html_x( 'Subscription URL (Admin)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_REFUND_AMOUNT'               => array(
				'name'      => esc_html_x( 'Refund amount', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_REFUND_AMOUNT',
				'tokenName' => esc_html_x( 'Refund amount', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_REFUND_REASON'               => array(
				'name'      => esc_html_x( 'Refund reason', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_REFUND_REASON',
				'tokenName' => esc_html_x( 'Refund reason', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PAYMENT_ID'                  => array(
				'name'      => esc_html_x( 'Payment ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_PAYMENT_ID',
				'tokenName' => esc_html_x( 'Payment ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_ID'                 => array(
				'name'      => esc_html_x( 'Customer ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_ID',
				'tokenName' => esc_html_x( 'Customer ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_EMAIL'              => array(
				'name'      => esc_html_x( 'Customer email', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_EMAIL',
				'tokenName' => esc_html_x( 'Customer email', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_NAME'               => array(
				'name'      => esc_html_x( 'Customer name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_NAME',
				'tokenName' => esc_html_x( 'Customer name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_STATUS'             => array(
				'name'      => esc_html_x( 'Customer status', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_STATUS',
				'tokenName' => esc_html_x( 'Customer status', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_DATE_CREATED'       => array(
				'name'      => esc_html_x( 'Customer date created', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_DATE_CREATED',
				'tokenName' => esc_html_x( 'Customer date created', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_USER_ID'            => array(
				'name'      => esc_html_x( 'Customer user ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_USER_ID',
				'tokenName' => esc_html_x( 'Customer user ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_PURCHASE_COUNT'     => array(
				'name'      => esc_html_x( 'Customer purchase count', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_PURCHASE_COUNT',
				'tokenName' => esc_html_x( 'Customer purchase count', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_PURCHASE_VALUE'     => array(
				'name'      => esc_html_x( 'Customer lifetime value', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_PURCHASE_VALUE',
				'tokenName' => esc_html_x( 'Customer lifetime value', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( empty( $hook_args ) ) {
			return false;
		}

		$order_id     = $hook_args[0];
		$order_detail = edd_get_payment( $order_id );

		if ( empty( $order_detail ) ) {
			return false;
		}

		$user_id = edd_get_payment_user_id( $order_detail->ID );

		$this->set_user_id( $user_id );

		// Check if this is a subscription payment.
		$cart_details = edd_get_payment_meta_cart_details( $order_id );

		if ( empty( $cart_details ) ) {
			return false;
		}

		// Check if download and price option match the trigger requirements.
		if ( ! isset( $trigger['meta'][ $this->trigger_meta ] ) ) {
			return false;
		}

		$selected_download     = $trigger['meta'][ $this->trigger_meta ];
		$selected_price_option = $trigger['meta'][ $this->price_option_meta ];

		$download_matches     = false;
		$price_option_matches = false;

		foreach ( $cart_details as $item ) {
			$download_id     = $item['id'];
			$price_option_id = isset( $item['item_number']['options']['price_id'] ) ? $item['item_number']['options']['price_id'] : null;

			// Check if download matches (allow "Any download" option (-1) or specific download).
			if ( intval( '-1' ) === intval( $selected_download ) || absint( $selected_download ) === absint( $download_id ) ) {
				$download_matches = true;
			}

			// Check if price option matches (allow "Any price option" option (-1) or specific price option).
			if ( intval( '-1' ) === intval( $selected_price_option ) || absint( $selected_price_option ) === absint( $price_option_id ) ) {
				$price_option_matches = true;
			}

			// If both match, store the matched IDs and break early.
			if ( $download_matches && $price_option_matches ) {
				$this->matched_download_id     = $download_id;
				$this->matched_price_option_id = $price_option_id;
				break;
			}
		}

		// Both download and price option must match.
		if ( ! $download_matches || ! $price_option_matches ) {
			return false;
		}

		return true;
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		$order_id = $hook_args[0];

		// Validate order exists
		$order_detail = edd_get_payment( $order_id );
		if ( ! $order_detail ) {
			return array();
		}

		// Use the matched IDs from validation instead of re-parsing cart details.
		$download_id     = $this->matched_download_id;
		$price_option_id = $this->matched_price_option_id;

		// If no matched IDs were stored, something went wrong in validation.
		if ( empty( $download_id ) ) {
			return array();
		}

		$download_post = get_post( $download_id );

		// Get price option name.
		$price_option_name = '';
		if ( null !== $price_option_id ) {
			$price_option_name = edd_get_price_option_name( $download_id, $price_option_id );
		}

		// Get featured image.
		$featured_image_id  = get_post_thumbnail_id( $download_id );
		$featured_image_url = $featured_image_id ? wp_get_attachment_url( $featured_image_id ) : '';

		// Get order URL (Admin).
		$order_url = edd_get_admin_url(
			array(
				'page' => 'edd-payment-history',
				'view' => 'view-order-details',
				'id'   => absint( $order_id ),
			)
		);

		// Get order URL (User) - receipt page.
		$order_url_user = edd_get_receipt_page_uri( $order_id );

		// Get subscription information.
		$subscription_id   = '';
		$subscription_url  = '';
		$recurring_period  = '';
		$signup_fee        = '';
		$times             = '';
		$free_trial_period = '';

		// Get the order object to find related subscriptions.
		$order = edd_get_order( $order_id );
		if ( $order ) {
			$subscriptions = edd_recurring_get_order_subscriptions( $order );

			if ( ! empty( $subscriptions ) && is_array( $subscriptions ) ) {
				// Get the first subscription (there should typically be only one for a specific product)
				$subscription_object = $subscriptions[0];
				$subscription_id     = $subscription_object->id;

				$subscription_url = edd_get_admin_url(
					array(
						'page' => 'edd-subscriptions',
						'id'   => absint( $subscription_id ),
					)
				);

				$recurring_period  = $subscription_object->period;
				$signup_fee        = number_format( $subscription_object->initial_amount - $subscription_object->recurring_amount, 2 );
				$times             = $subscription_object->bill_times;
				$free_trial_period = $subscription_object->trial_period;
			}
		}

		// Get total discount from order details.
		$total_discount = 0;
		if ( isset( $order_detail->discounts ) && is_numeric( $order_detail->discounts ) ) {
			$total_discount = $order_detail->discounts;
		}

		// Get refund information.
		$refund_amount = '';
		$refund_reason = '';

		// Check if this payment has refunds.
		$refunds = edd_get_order_refunds( $order_id );

		if ( ! empty( $refunds ) ) {
			$latest_refund = end( $refunds );
			$refund_amount = number_format( abs( $latest_refund->total ), 2 );
			$refund_reason = edd_get_order_meta( $latest_refund->id, 'reason', true );
			$refund_reason = $refund_reason ? $refund_reason : '';
		}

		// Get customer information.
		$customer_id             = '';
		$customer_email          = '';
		$customer_name           = '';
		$customer_status         = '';
		$customer_date_created   = '';
		$customer_user_id        = '';
		$customer_purchase_count = '';
		$customer_purchase_value = '';

		if ( ! empty( $order_detail->customer_id ) ) {
			$customer = edd_get_customer( $order_detail->customer_id );
			if ( $customer ) {
				$customer_id             = $customer->id;
				$customer_email          = $customer->email;
				$customer_name           = $customer->name;
				$customer_status         = $customer->status;
				$customer_date_created   = $customer->date_created;
				$customer_user_id        = $customer->user_id;
				$customer_purchase_count = $customer->purchase_count;
				$customer_purchase_value = number_format( $customer->purchase_value, 2 );
			}
		}

		$tokens = array(
			'EDD_DOWNLOAD_TITLE'              => $download_post ? $download_post->post_title : '',
			'EDD_DOWNLOAD_ID'                 => $download_id,
			'EDD_DOWNLOAD_URL'                => get_permalink( $download_id ),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => $featured_image_id,
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => $featured_image_url,
			'EDD_PRICE_OPTION_NAME'           => $price_option_name,
			'EDD_PRICE_OPTION_ID'             => $price_option_id,
			'EDD_ORDER_ID'                    => $order_id,
			'EDD_ORDER_URL'                   => $order_url,
			'EDD_ORDER_URL_USER'              => $order_url_user,
			'EDD_ORDER_DISCOUNTS'             => number_format( $total_discount, 2 ),
			'EDD_ORDER_SUBTOTAL'              => number_format( $order_detail->subtotal, 2 ),
			'EDD_ORDER_TAX'                   => number_format( $order_detail->tax, 2 ),
			'EDD_ORDER_TOTAL'                 => number_format( $order_detail->total, 2 ),
			'EDD_PAYMENT_METHOD'              => $order_detail->gateway,
			'EDD_DISCOUNT_CODES'              => $order_detail->discounts,
			'EDD_RECURRING_PERIOD'            => $recurring_period,
			'EDD_SIGNUP_FEE'                  => $signup_fee,
			'EDD_TIMES'                       => $times,
			'EDD_FREE_TRIAL_PERIOD'           => $free_trial_period,
			'EDD_SUBSCRIPTION_ID'             => $subscription_id,
			'EDD_SUBSCRIPTION_URL'            => $subscription_url,
			'EDD_REFUND_AMOUNT'               => $refund_amount,
			'EDD_REFUND_REASON'               => $refund_reason,
			'EDD_PAYMENT_ID'                  => $order_id,
			'EDD_CUSTOMER_ID'                 => $customer_id,
			'EDD_CUSTOMER_EMAIL'              => $customer_email,
			'EDD_CUSTOMER_NAME'               => $customer_name,
			'EDD_CUSTOMER_STATUS'             => $customer_status,
			'EDD_CUSTOMER_DATE_CREATED'       => $customer_date_created,
			'EDD_CUSTOMER_USER_ID'            => $customer_user_id,
			'EDD_CUSTOMER_PURCHASE_COUNT'     => $customer_purchase_count,
			'EDD_CUSTOMER_PURCHASE_VALUE'     => $customer_purchase_value,
		);

		return $tokens;
	}
}
