<?php

namespace Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class EDD_USER_CANCELS_SUBSCRIPTION_PRODWITHPRICEOPTION
 *
 * @package Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration
 * @method \Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration\Edd_Recurring_Pro_Helpers get_item_helpers()
 */
class EDD_USER_CANCELS_SUBSCRIPTION_PRODWITHPRICEOPTION extends Trigger {

	/**
	 * Trigger code.
	 *
	 * @var string
	 */
	protected $trigger_code = 'EDD_USER_CANCELS_SUBSCRIPTION_PRODWITHPRICEOPTION';

	/**
	 * Trigger meta.
	 *
	 * @var string
	 */
	protected $trigger_meta = 'EDDPRODUCT';

	/**
	 * Price option meta.
	 *
	 * @var string
	 */
	protected $price_option_meta = 'EDDPRICEOPTION';


	/**
	 * Set up Automator trigger.
	 */
	protected function setup_trigger() {
		$this->set_integration( 'EDD_RECURRING' );
		$this->set_trigger_code( $this->trigger_code );
		$this->set_trigger_meta( $this->trigger_meta );
		$this->set_is_pro( true );
		$this->set_trigger_type( 'user' );
		$this->set_is_login_required( false );
		$this->set_support_link( Automator()->get_author_support_link( $this->trigger_code, 'integration/easy-digital-downloads/' ) );
		// translators: %1$s: download, %2$s: price option
		$this->set_sentence( sprintf( esc_html_x( 'A user cancels their subscription to {{a price option:%2$s}} of {{a download:%1$s}}', 'EDD - Recurring Payments', 'uncanny-automator-pro' ), $this->get_trigger_meta(), $this->price_option_meta . ':' . $this->get_trigger_meta() ) );
		$this->set_readable_sentence( esc_html_x( 'A user cancels their subscription to {{a price option}} of {{a download}}', 'EDD - Recurring Payments', 'uncanny-automator-pro' ) );
		$this->add_action( 'edd_subscription_cancelled', 10, 2 );
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			$this->get_item_helpers()->all_recurring_edd_downloads( esc_html_x( 'Download', 'EDD - Recurring Payments', 'uncanny-automator-pro' ), $this->get_trigger_meta() ),
			array(
				'option_code'     => $this->price_option_meta,
				'label'           => esc_html_x( 'Price option', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'input_type'      => 'select',
				'required'        => true,
				'options'         => array(),
				'relevant_tokens' => array(),
				'ajax'            => array(
					'event'         => 'parent_fields_change',
					'endpoint'      => 'automator_edd_price_options_handler',
					'listen_fields' => array( $this->get_trigger_meta() ),
				),
			),
		);
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger
	 * @param array $tokens
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		return array(
			'EDD_DOWNLOAD_TITLE'              => array(
				'name'      => esc_html_x( 'Download title', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_DOWNLOAD_TITLE',
				'tokenName' => esc_html_x( 'Download title', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_ID'                 => array(
				'name'      => esc_html_x( 'Download ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_ID',
				'tokenName' => esc_html_x( 'Download ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_URL'                => array(
				'name'      => esc_html_x( 'Download URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_URL',
				'tokenName' => esc_html_x( 'Download URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => array(
				'name'      => esc_html_x( 'Download featured image ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_ID',
				'tokenName' => esc_html_x( 'Download featured image ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => array(
				'name'      => esc_html_x( 'Download featured image URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_DOWNLOAD_FEATURED_IMAGE_URL',
				'tokenName' => esc_html_x( 'Download featured image URL', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PRICE_OPTION_NAME'           => array(
				'name'      => esc_html_x( 'Price option name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PRICE_OPTION_NAME',
				'tokenName' => esc_html_x( 'Price option name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PRICE_OPTION_ID'             => array(
				'name'      => esc_html_x( 'Price option ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_PRICE_OPTION_ID',
				'tokenName' => esc_html_x( 'Price option ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PAYMENT_METHOD'              => array(
				'name'      => esc_html_x( 'Payment method', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PAYMENT_METHOD',
				'tokenName' => esc_html_x( 'Payment method', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_RECURRING_PERIOD'            => array(
				'name'      => esc_html_x( 'Recurring period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_RECURRING_PERIOD',
				'tokenName' => esc_html_x( 'Recurring period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_SIGNUP_FEE'                  => array(
				'name'      => esc_html_x( 'Signup fee', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_SIGNUP_FEE',
				'tokenName' => esc_html_x( 'Signup fee', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_TIMES'                       => array(
				'name'      => esc_html_x( 'Times', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_TIMES',
				'tokenName' => esc_html_x( 'Times', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_FREE_TRIAL_PERIOD'           => array(
				'name'      => esc_html_x( 'Free trial period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_FREE_TRIAL_PERIOD',
				'tokenName' => esc_html_x( 'Free trial period', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_SUBSCRIPTION_ID'             => array(
				'name'      => esc_html_x( 'Subscription ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_SUBSCRIPTION_ID',
				'tokenName' => esc_html_x( 'Subscription ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_SUBSCRIPTION_URL'            => array(
				'name'      => esc_html_x( 'Subscription URL (Admin)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'url',
				'tokenId'   => 'EDD_SUBSCRIPTION_URL',
				'tokenName' => esc_html_x( 'Subscription URL (Admin)', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_PREVIOUS_STATUS'             => array(
				'name'      => esc_html_x( 'Previous status', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_PREVIOUS_STATUS',
				'tokenName' => esc_html_x( 'Previous status', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_ID'                 => array(
				'name'      => esc_html_x( 'Customer ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_ID',
				'tokenName' => esc_html_x( 'Customer ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_EMAIL'              => array(
				'name'      => esc_html_x( 'Customer email', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_EMAIL',
				'tokenName' => esc_html_x( 'Customer email', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_NAME'               => array(
				'name'      => esc_html_x( 'Customer name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_NAME',
				'tokenName' => esc_html_x( 'Customer name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_STATUS'             => array(
				'name'      => esc_html_x( 'Customer status', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_STATUS',
				'tokenName' => esc_html_x( 'Customer status', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_DATE_CREATED'       => array(
				'name'      => esc_html_x( 'Customer date created', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_DATE_CREATED',
				'tokenName' => esc_html_x( 'Customer date created', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_USER_ID'            => array(
				'name'      => esc_html_x( 'Customer user ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_USER_ID',
				'tokenName' => esc_html_x( 'Customer user ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_PURCHASE_COUNT'     => array(
				'name'      => esc_html_x( 'Customer purchase count', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'int',
				'tokenId'   => 'EDD_CUSTOMER_PURCHASE_COUNT',
				'tokenName' => esc_html_x( 'Customer purchase count', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
			'EDD_CUSTOMER_PURCHASE_VALUE'     => array(
				'name'      => esc_html_x( 'Customer lifetime value', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type'      => 'text',
				'tokenId'   => 'EDD_CUSTOMER_PURCHASE_VALUE',
				'tokenName' => esc_html_x( 'Customer lifetime value', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( empty( $hook_args ) || count( $hook_args ) < 2 ) {
			return false;
		}

		if ( ! isset( $trigger['meta'][ $this->trigger_meta ] ) ) {
			return false;
		}

		$selected_download     = $trigger['meta'][ $this->trigger_meta ];
		$selected_price_option = $trigger['meta'][ $this->price_option_meta ];

		// Fallback: try to get price option from different possible keys.
		if ( empty( $selected_price_option ) ) {
			if ( isset( $trigger['meta']['EDDPRICEOPTION'] ) ) {
				$selected_price_option = $trigger['meta']['EDDPRICEOPTION'];
			} elseif ( isset( $trigger['meta'][ 'EDDPRICEOPTION:' . $this->get_trigger_meta() ] ) ) {
				$selected_price_option = $trigger['meta'][ 'EDDPRICEOPTION:' . $this->get_trigger_meta() ];
			}
		}

		$subscription_object = $hook_args[1];
		$download_id         = $subscription_object->product_id;
		$price_option_id     = $subscription_object->price_id;

		// Set user ID for the trigger from customer data
		$user_id = 0;
		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = \edd_get_customer( $subscription_object->customer_id );
			if ( $customer && ! empty( $customer->user_id ) ) {
				$user_id = $customer->user_id;
			}
		}

		// Fallback to current user ID if no customer user ID found
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( $user_id ) {
			$this->set_user_id( $user_id );
		}

		// Check if download matches (allow "Any download" option (-1) or specific download).
		$download_matches = ( intval( '-1' ) === intval( $selected_download ) || absint( $selected_download ) === absint( $download_id ) );

		// Check if price option matches (allow "Any price option" option (-1) or specific price option).
		$price_option_matches = ( intval( '-1' ) === intval( $selected_price_option ) || absint( $selected_price_option ) === absint( $price_option_id ) );

		// Both download and price option must match.
		return $download_matches && $price_option_matches;
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		if ( empty( $hook_args ) || count( $hook_args ) < 2 ) {
			return array();
		}

		$subscription_object = $hook_args[1];
		$download_id         = $subscription_object->product_id;
		$price_option_id     = $subscription_object->price_id;

		$download_post = get_post( $download_id );

		// Get price option name.
		$price_option_name = '';
		if ( null !== $price_option_id ) {
			$price_option_name = edd_get_price_option_name( $download_id, $price_option_id );
		}

		// Get featured image.
		$featured_image_id  = get_post_thumbnail_id( $download_id );
		$featured_image_url = $featured_image_id ? wp_get_attachment_url( $featured_image_id ) : '';

		// Get subscription URL (Admin).
		$subscription_url = edd_get_admin_url(
			array(
				'page' => 'edd-subscriptions',
				'id'   => absint( $subscription_object->id ),
			)
		);

		// Get payment information.
		$payment_method = '';
		if ( ! empty( $subscription_object->parent_payment_id ) ) {
			$payment        = edd_get_payment( $subscription_object->parent_payment_id );
			$payment_method = $payment ? $payment->gateway : '';
		}

		// Get subscription details.
		$recurring_period  = $subscription_object->period;
		$signup_fee        = number_format( $subscription_object->initial_amount - $subscription_object->recurring_amount, 2 );
		$times             = $subscription_object->bill_times;
		$free_trial_period = $subscription_object->trial_period;

		// Get previous status (before cancellation).
		$previous_status = '';
		if ( isset( $subscription_object->status ) ) {
			$previous_status = $subscription_object->status;
		}

		// Get customer information.
		$customer_id             = '';
		$customer_email          = '';
		$customer_name           = '';
		$customer_status         = '';
		$customer_date_created   = '';
		$customer_user_id        = '';
		$customer_purchase_count = '';
		$customer_purchase_value = '';

		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = edd_get_customer( $subscription_object->customer_id );
			if ( $customer ) {
				$customer_id             = $customer->id;
				$customer_email          = $customer->email;
				$customer_name           = $customer->name;
				$customer_status         = $customer->status;
				$customer_date_created   = $customer->date_created;
				$customer_user_id        = $customer->user_id;
				$customer_purchase_count = $customer->purchase_count;
				$customer_purchase_value = number_format( $customer->purchase_value, 2 );
			}
		}

		$tokens = array(
			'EDD_DOWNLOAD_TITLE'              => $download_post ? $download_post->post_title : '',
			'EDD_DOWNLOAD_ID'                 => $download_id,
			'EDD_DOWNLOAD_URL'                => get_permalink( $download_id ),
			'EDD_DOWNLOAD_FEATURED_IMAGE_ID'  => $featured_image_id,
			'EDD_DOWNLOAD_FEATURED_IMAGE_URL' => $featured_image_url,
			'EDD_PRICE_OPTION_NAME'           => $price_option_name,
			'EDD_PRICE_OPTION_ID'             => $price_option_id,
			'EDD_PAYMENT_METHOD'              => $payment_method,
			'EDD_RECURRING_PERIOD'            => $recurring_period,
			'EDD_SIGNUP_FEE'                  => $signup_fee,
			'EDD_TIMES'                       => $times,
			'EDD_FREE_TRIAL_PERIOD'           => $free_trial_period,
			'EDD_SUBSCRIPTION_ID'             => $subscription_object->id,
			'EDD_SUBSCRIPTION_URL'            => $subscription_url,
			'EDD_PREVIOUS_STATUS'             => $previous_status,
			'EDD_CUSTOMER_ID'                 => $customer_id,
			'EDD_CUSTOMER_EMAIL'              => $customer_email,
			'EDD_CUSTOMER_NAME'               => $customer_name,
			'EDD_CUSTOMER_STATUS'             => $customer_status,
			'EDD_CUSTOMER_DATE_CREATED'       => $customer_date_created,
			'EDD_CUSTOMER_USER_ID'            => $customer_user_id,
			'EDD_CUSTOMER_PURCHASE_COUNT'     => $customer_purchase_count,
			'EDD_CUSTOMER_PURCHASE_VALUE'     => $customer_purchase_value,
		);

		return $tokens;
	}
}
