<?php

namespace Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration;

/**
 * Class EDD_SET_SUBSCRIPTION_EXPIRATION_DATE
 *
 * @package Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration
 * @method \Uncanny_Automator_Pro\Integrations\Edd_Recurring_Integration\Edd_Recurring_Pro_Helpers get_item_helpers()
 */
class EDD_SET_SUBSCRIPTION_EXPIRATION_DATE extends \Uncanny_Automator\Recipe\Action {

	/**
	 * @return mixed|void
	 */
	protected function setup_action() {

		$this->set_integration( 'EDD_RECURRING' );
		$this->set_action_code( 'EDDR_SET_SUBSCRIPTION_EXPIRY' );
		$this->set_action_meta( 'EDDR_PRODUCTS' );
		$this->set_is_pro( true );
		$this->set_requires_user( true );
		$this->set_sentence(
			sprintf(
			// translators: %1$s: Subscription download
			// translators: %2$s: Expiration date
				esc_html_x( 'Set {{a subscription download:%1$s}} to expire on {{a specific date:%2$s}} for the user', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				$this->get_action_meta(),
				'EXPIRATION_DATE:' . $this->get_action_meta()
			)
		);
		$this->set_readable_sentence( esc_html_x( 'Set {{a subscription download}} to expire on {{a specific date}} for the user', 'EDD - Recurring Payments', 'uncanny-automator-pro' ) );
	}

	/**
	 * Define the Action's options
	 *
	 * @return void
	 */
	public function options() {

		return array(
			$this->get_item_helpers()->all_recurring_edd_downloads( esc_html_x( 'Download', 'EDD - Recurring Payments', 'uncanny-automator-pro' ), $this->get_action_meta() ),
			array(
				'input_type'      => 'date',
				'option_code'     => 'EXPIRATION_DATE',
				'label'           => esc_html_x( 'Expiration date', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'relevant_tokens' => array(),
			),
		);
	}

	/**
	 * @return array[]
	 */
	public function define_tokens() {
		return array(
			'SUBSCRIPTION_ID' => array(
				'name' => esc_html_x( 'Subscription ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type' => 'int',
			),
			'DOWNLOAD_ID'     => array(
				'name' => esc_html_x( 'Download ID', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type' => 'int',
			),
			'DOWNLOAD_NAME'   => array(
				'name' => esc_html_x( 'Download name', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'EXPIRATION_DATE' => array(
				'name' => esc_html_x( 'Expiration date', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
				'type' => 'datetime',
			),
		);
	}

	/**
	 * @param int   $user_id
	 * @param array $action_data
	 * @param int   $recipe_id
	 * @param array $args
	 * @param       $parsed
	 */
	protected function process_action( $user_id, $action_data, $recipe_id, $args, $parsed ) {
		// Get the selected product ID
		$download_id     = sanitize_text_field( $parsed[ $this->get_action_meta() ] );
		$expiration_date = sanitize_text_field( $parsed['EXPIRATION_DATE'] );

		if ( empty( $download_id ) ) {
			$this->add_log_error( esc_html_x( 'Invalid download ID.', 'EDD - Recurring Payments', 'uncanny-automator-pro' ) );

			return false;
		}

		if ( empty( $expiration_date ) ) {
			$this->add_log_error( esc_html_x( 'Please enter expiry date.', 'EDD - Recurring Payments', 'uncanny-automator-pro' ) );

			return false;
		}

		$download_name = sanitize_text_field( $parsed[ $this->get_action_meta() . '_readable' ] );
		$subscriber    = new \EDD_Recurring_Subscriber( $user_id, true );
		$subscriptions = $subscriber->get_subscriptions( $download_id, array( 'active', 'trialling' ) );
		if ( empty( $subscriptions ) ) {
			$this->add_log_error(
				sprintf(
					// translators: %1$s: Download name
					esc_html_x( 'The user does not have any active subscription to "%1$s".', 'EDD - Recurring Payments', 'uncanny-automator-pro' ),
					$download_name
				)
			);

			return false;
		}

		foreach ( $subscriptions as $subscription ) {
			$subs = new \EDD_Subscription( $subscription->id );
			$subs->update( array( 'expiration' => gmdate( 'Y-m-d H:i:s', strtotime( $expiration_date ) ) ) );

			$this->hydrate_tokens(
				array(
					'SUBSCRIPTION_ID' => $subscription->id,
					'DOWNLOAD_ID'     => $download_id,
					'DOWNLOAD_NAME'   => $download_name,
					'EXPIRATION_DATE' => $subs->expiration,
				)
			);
			break;
		}

		return true;
	}
}
