<?php

namespace Uncanny_Automator_Pro\Integrations\Asana;

/**
 * Asana Pro Tokens Class
 *
 * Provides common token definitions and hydration methods for Asana triggers
 * to eliminate code duplication across trigger files.
 *
 * @package Uncanny_Automator_Pro\Integrations\Asana\Tokens
 */
class Asana_Pro_Tokens {

	/**
	 * Get workspace token definitions.
	 *
	 * @return array
	 */
	public static function get_workspace_token_definitions() {
		return array(
			array(
				'tokenId'   => 'ASANA_WORKSPACE_ID',
				'tokenName' => esc_html_x( 'Workspace ID', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_WORKSPACE_NAME',
				'tokenName' => esc_html_x( 'Workspace name', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Hydrate workspace token values.
	 *
	 * @param array $event_data The event data from webhook.
	 *
	 * @return array
	 */
	public static function hydrate_workspace_tokens( $event_data ) {
		return array(
			'ASANA_WORKSPACE_ID'   => $event_data['workspace_id'] ?? '',
			'ASANA_WORKSPACE_NAME' => $event_data['workspace_name'] ?? '',
		);
	}

	/**
	 * Get project token definitions.
	 *
	 * @return array
	 */
	public static function get_project_token_definitions() {
		return array(
			array(
				'tokenId'   => 'ASANA_PROJECT_ID',
				'tokenName' => esc_html_x( 'Project ID', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_PROJECT_NAME',
				'tokenName' => esc_html_x( 'Project name', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Hydrate project token values.
	 *
	 * @param array $event_data The event data from webhook.
	 * @param string $project_id The project ID.
	 *
	 * @return array
	 */
	public static function hydrate_project_tokens( $event_data, $project_id ) {
		return array(
			'ASANA_PROJECT_ID'   => $project_id,
			'ASANA_PROJECT_NAME' => $event_data['project_name'] ?? '',
		);
	}

	/**
	 * Get basic task token definitions (core task properties).
	 *
	 * @return array
	 */
	public static function get_basic_task_token_definitions() {
		return array(
			array(
				'tokenId'   => 'ASANA_TASK_ID',
				'tokenName' => esc_html_x( 'Task ID', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_TASK_NAME',
				'tokenName' => esc_html_x( 'Task name', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_TASK_TYPE',
				'tokenName' => esc_html_x( 'Task type', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_TASK_STATUS',
				'tokenName' => esc_html_x( 'Task status', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_TASK_URL',
				'tokenName' => esc_html_x( 'Task URL', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);
	}

	/**
	 * Hydrate basic task token values.
	 *
	 * @param array $event_data The event data from webhook.
	 * @param array $task_details Optional task details from API call.
	 *
	 * @return array
	 */
	public static function hydrate_basic_task_tokens( $event_data, $task_details = null ) {
		$task_details = is_array( $task_details ) ? $task_details : array();

		return array(
			'ASANA_TASK_ID'     => $event_data['task_id'] ?? '',
			'ASANA_TASK_NAME'   => $task_details['name'] ?? '',
			'ASANA_TASK_TYPE'   => $task_details['resource_subtype'] ?? '',
			'ASANA_TASK_STATUS' => ! empty( $task_details['completed'] ) ? 'completed' : 'incomplete',
			'ASANA_TASK_URL'    => $task_details['permalink_url'] ?? '',
		);
	}

	/**
	 * Get extended task token definitions (basic + descriptions).
	 *
	 * @return array
	 */
	public static function get_extended_task_token_definitions() {
		return array_merge(
			self::get_basic_task_token_definitions(),
			array(
				array(
					'tokenId'   => 'ASANA_TASK_DESCRIPTION',
					'tokenName' => esc_html_x( 'Task description', 'Asana', 'uncanny-automator-pro' ),
					'tokenType' => 'text',
				),
				array(
					'tokenId'   => 'ASANA_TASK_DESCRIPTION_HTML',
					'tokenName' => esc_html_x( 'Task description (HTML)', 'Asana', 'uncanny-automator-pro' ),
					'tokenType' => 'text',
				),
			)
		);
	}

	/**
	 * Hydrate extended task token values.
	 *
	 * @param array $event_data The event data from webhook.
	 * @param array $task_details Optional task details from API call.
	 *
	 * @return array
	 */
	public static function hydrate_extended_task_tokens( $event_data, $task_details = null ) {
		$task_details = is_array( $task_details ) ? $task_details : array();

		$tokens = self::hydrate_basic_task_tokens( $event_data, $task_details );

		// Add extended task tokens (descriptions)
		$tokens['ASANA_TASK_DESCRIPTION']      = $task_details['notes'] ?? '';
		$tokens['ASANA_TASK_DESCRIPTION_HTML'] = $task_details['html_notes'] ?? '';

		return $tokens;
	}

	/**
	 * Get task assignee token definitions.
	 *
	 * @return array
	 */
	public static function get_task_assignee_token_definitions() {
		return array(
			array(
				'tokenId'   => 'ASANA_TASK_ASSIGNEE_ID',
				'tokenName' => esc_html_x( 'Task assignee member ID', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_TASK_ASSIGNEE_NAME',
				'tokenName' => esc_html_x( 'Task assignee member name', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'ASANA_TASK_ASSIGNEE_EMAIL',
				'tokenName' => esc_html_x( 'Task assignee member email', 'Asana', 'uncanny-automator-pro' ),
				'tokenType' => 'email',
			),
		);
	}

	/**
	 * Hydrate task assignee token values.
	 *
	 * @param array $task_details Task details from API call.
	 *
	 * @return array
	 */
	public static function hydrate_task_assignee_tokens( $task_details ) {
		$task_details = is_array( $task_details ) ? $task_details : array();
		$assignee     = $task_details['assignee'] ?? array();
		return array(
			'ASANA_TASK_ASSIGNEE_ID'    => $assignee['gid'] ?? '',
			'ASANA_TASK_ASSIGNEE_NAME'  => $assignee['name'] ?? '',
			'ASANA_TASK_ASSIGNEE_EMAIL' => $assignee['email'] ?? '',
		);
	}
}
