<?php

namespace Uncanny_Automator_Pro\Integrations\Asana;

use Uncanny_Automator\App_Integrations\App_Webhooks;
use Exception;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Class Asana_Pro_Webhooks
 *
 * @package Uncanny_Automator
 *
 * @property Asana_Pro_Api_Caller $api
 * @property Asana_Pro_App_Helpers $helpers
 */
class Asana_Pro_Webhooks extends App_Webhooks {

	/**
	 * The option name for the webhooks config.
	 *
	 * @var string
	 */
	const MANAGER_OPTION = 'asana_webhooks_manager';

	////////////////////////////////////////////////////////////
	// Abstract overrides
	////////////////////////////////////////////////////////////

	/**
	 * Validate the webhook.
	 * Override to handle Asana-specific validation (signature).
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return bool|WP_REST_Response
	 * @throws Exception If the webhook is invalid.
	 */
	protected function validate_webhook( $request ) {

		// Check if this is a handshake request.
		$secret = $request->get_header( 'x_hook_secret' );
		if ( ! empty( $secret ) ) {
			return $this->handle_webhook_creation_handshake( $secret );
		}

		// Validate we have data.
		$data = $request->get_json_params();
		if ( empty( $data ) ) {
			throw new Exception( esc_html_x( 'Webhook not accepted', 'Asana', 'uncanny-automator-pro' ) );
		}

		// Retrieve the project ID from the URL param: pid=<project_gid>
		$project_id     = $request->get_param( 'pid' );
		$project_config = ! empty( $project_id ) ? $this->get_project_config( $project_id ) : null;

		if ( empty( $project_config ) ) {
			throw new Exception( esc_html_x( 'Project is not configured for webhooks on this site', 'Asana', 'uncanny-automator-pro' ) );
		}

		// Validate the webhook signature.
		$secret    = $project_config['secret'] ?? null;
		$signature = $request->get_header( 'X-Hook-Signature' );
		if ( empty( $secret ) || ! $this->validate_asana_signature( $request, $secret ) ) {
			throw new Exception( esc_html_x( 'Webhook verification failed', 'Asana', 'uncanny-automator-pro' ) );
		}

		return true;
	}

	/**
	 * Process webhook request.
	 * Override to handle Asana-specific event processing with deduplication.
	 *
	 * @param string $action_name   The action name.
	 * @param array  $action_params The action parameters array.
	 *
	 * @return void
	 */
	protected function process_webhook_request( $action_name, $action_params ) {
		// Get the webhook data from the current request.
		$data       = $action_params[0] ?? null;
		$project_id = $data['pid'] ?? null;

		if ( ! empty( $data ) ) {
			// Process the webhook events with our custom logic.
			$this->process_webhook_events( $data, $project_id );
		}
	}

	////////////////////////////////////////////////////////////
	// Webhook validation and processing
	////////////////////////////////////////////////////////////

	/**
	 * Handle the webhook creation handshake.
	 *
	 * @param string $secret The webhook secret.
	 *
	 * @return WP_REST_Response
	 */
	private function handle_webhook_creation_handshake( $secret ) {
		$response = new WP_REST_Response();
		$response->set_status( 200 );
		$response->header( 'X-Hook-Secret', $secret );
		return $response;
	}

	/**
	 * Validate the Asana webhook signature.
	 * Uses X-Hook-Signature header and HMAC SHA256 as per Asana docs.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @param string $secret The webhook secret (X-Hook-Secret from handshake).
	 *
	 * @return bool
	 */
	private function validate_asana_signature( $request, $secret ) {
		$signature = $request->get_header( 'X-Hook-Signature' );
		if ( empty( $signature ) ) {
			return false;
		}

		$payload            = $request->get_body();
		$expected_signature = hash_hmac( 'sha256', $payload, $secret );

		return hash_equals( $expected_signature, $signature );
	}

	/**
	 * Process webhook events for the project.
	 * Groups task events to prevent rapid-fire triggers.
	 *
	 * @param array $data The webhook data.
	 * @param string $project_id The project ID.
	 */
	private function process_webhook_events( $data, $project_id ) {
		if ( empty( $data['events'] ) || ! is_array( $data['events'] ) ) {
			return;
		}

		if ( empty( $project_id ) ) {
			return;
		}

		$project_config = $this->get_project_config( $project_id );
		if ( empty( $project_config ) ) {
			return;
		}

		// Use dedicated processor for clean separation of concerns
		$processor = new Asana_Webhook_Processor();
		$processor->process_events( $data['events'], $project_config );
	}

	////////////////////////////////////////////////////////////
	// Webhook manager
	////////////////////////////////////////////////////////////

	/**
	 * Get the webhooks config.
	 *
	 * @return array
	 */
	public function get_webhook_manager_config() {
		$config = automator_get_option( self::MANAGER_OPTION, array() );

		if ( empty( $config ) ) {
			// Get user workspaces first.
			$workspaces = $this->api->get_user_workspaces();
			if ( empty( $workspaces ) ) {
				return array();
			}

			$config = array();

			// Get projects from each workspace.
			foreach ( $workspaces as $workspace ) {
				$projects = $this->api->get_workspace_projects( $workspace['value'] );
				if ( ! empty( $projects ) ) {
					foreach ( $projects as $project ) {
						// Add project config.
						$config[ $project['value'] ] = array(
							'id'             => $project['value'],
							'name'           => $project['text'],
							'workspace_id'   => $workspace['value'],
							'workspace_name' => $workspace['text'],
							'hook_id'        => null, // Will get set when generated.
							'endpoint'       => $this->get_webhook_endpoint(),
							'secret'         => null, // Will get set when generated.
							'events'         => array(),
							'url'            => add_query_arg(
								array( 'pid' => $project['value'] ),
								$this->get_webhook_url()
							),
							'connected_at'   => null,
						);
					}
				}
			}

			// Sort projects by workspace name first, then by project name.
			uasort(
				$config,
				function ( $a, $b ) {
					$workspace_compare = strcmp( $a['workspace_name'], $b['workspace_name'] );
					if ( 0 !== $workspace_compare ) {
						return $workspace_compare;
					}
					return strcmp( $a['name'], $b['name'] );
				}
			);

			// Update the option.
			automator_update_option( self::MANAGER_OPTION, $config );
		}

		return $config;
	}

	/**
	 * Get the project config by project ID.
	 *
	 * @param int $project_id The ID of the project.
	 *
	 * @return array
	 */
	public function get_project_config( $project_id ) {
		$config = $this->get_webhook_manager_config();
		return $config[ $project_id ] ?? array();
	}

	/**
	 * Update the project config.
	 *
	 * @param int $project_id The ID of the project.
	 * @param array $project The project config.
	 *
	 * @return void
	 */
	public function update_project_config( $project_id, $project ) {
		$config                = $this->get_webhook_manager_config();
		$config[ $project_id ] = $project;
		automator_update_option( self::MANAGER_OPTION, $config );
		$this->manage_webhooks_enabled_status( $config );
	}

	/**
	 * Manage the webhooks enabled status.
	 *
	 * @param array $config The webhook config.
	 *
	 * @return void
	 */
	private function manage_webhooks_enabled_status( $config ) {
		$enabled = false;
		foreach ( $config as $project ) {
			if ( ! empty( $project['hook_id'] ) ) {
				$enabled = true;
				break;
			}
		}

		// Update the integration status.
		$this->store_webhooks_enabled_status( $enabled );
	}

	/**
	 * Get user token data from webhook event data.
	 *
	 * @param array $event_data The event data from webhook.
	 *
	 * @return array User information array with value, text, and email keys.
	 */
	public function get_user_token_data( $event_data ) {
		$user_id      = $event_data['user_id'] ?? null;
		$workspace_id = $event_data['workspace_id'] ?? null;

		$user = array(
			'value' => $user_id,
			'text'  => '-',
			'email' => '-',
		);

		if ( ! empty( $user_id ) && ! empty( $workspace_id ) ) {
			$option        = $this->helpers->get_workspace_user_option( $workspace_id, $user_id );
			$user['text']  = $option['text'] ?? '-';
			$user['email'] = $option['email'] ?? '-';
		} else {
			$user['text'] = esc_html_x( 'System generated', 'Asana', 'uncanny-automator-pro' );
		}

		return $user;
	}
}
