<?php

namespace Uncanny_Automator_Pro;

/**
 * Class  AMELIA_USER_EVENT_BOOKING_STATUS_UPDATED
 * @package Uncanny_Automator_Pro
 */
class AMELIA_USER_EVENT_BOOKING_STATUS_UPDATED extends \Uncanny_Automator\Recipe\Trigger {

	/**
	 * @return mixed
	 */
	protected function setup_trigger() {
		$this->set_integration( 'AMELIABOOKING' );
		$this->set_trigger_code( 'AMELIA_EVENT_BOOKING_STATUSES' );
		$this->set_trigger_meta( 'AMELIA_EVENTS' );
		$this->set_helper( new Ameliabooking_Pro_Helpers() );
		$this->set_is_pro( true );
		$this->set_readable_sentence( esc_html_x( "A user's booking for {{an event}} changes to {{a specific status}}", 'Ameliabooking', 'uncanny-automator-pro' ) );
		/*translator: Amelia event booking status updated*/
		$this->set_sentence( sprintf( esc_html_x( "A user's booking for {{an event:%1\$s}} changes to {{a specific status:%2\$s}}", 'Ameliabooking', 'uncanny-automator-pro' ), $this->get_trigger_meta(), $this->get_trigger_meta() . '_BOOKING_STATUS:' . $this->get_trigger_meta() ) );
		$this->add_action( 'automator_amelia_event_booking_status_updated', 10, 3 );
	}

	/**
	 * options
	 *
	 * @return array
	 */
	public function options() {
		$events        = Automator()->helpers->recipe->ameliabooking->options->get_events_dropdown();
		$events_option = array();
		foreach ( $events as $k => $event ) {
			$events_option[] = array(
				'text'  => esc_html_x( $event, 'Ameliabooking', 'uncanny-automator-pro' ),
				'value' => $k,
			);
		}

		return array(
			array(
				'option_code'     => $this->get_trigger_meta(),
				'label'           => esc_html_x( 'Events', 'Ameliabooking', 'uncanny-automator-pro' ),
				'input_type'      => 'select',
				'required'        => true,
				'options'         => $events_option,
				'relevant_tokens' => array(),
			),
			array(
				'input_type'            => 'select',
				'option_code'           => $this->get_trigger_meta() . '_BOOKING_STATUS',
				'required'              => true,
				'label'                 => esc_html_x( 'Status', 'Ameliabooking', 'uncanny-automator-pro' ),
				'options'               => $this->get_helper()->get_event_booking_statuses( true ),
				'supports_custom_value' => false,
				'relevant_tokens'       => array(),
			),
		);
	}

	/**
	 * define_tokens
	 *
	 * Override this method if you want to add recipe-specific tokens such as form fields etc.
	 *
	 * @param mixed $tokens
	 * @param mixed $args
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		$trigger_tokens = array(
			array(
				'tokenId'   => 'EVENT_ID',
				'tokenName' => esc_html_x( 'Event ID', 'Ameliabooking', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'EVENT_TITLE',
				'tokenName' => esc_html_x( 'Event title', 'Ameliabooking', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EVENT_BOOKING_STATUS',
				'tokenName' => esc_html_x( 'Status', 'Ameliabooking', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
		);

		return array_merge( $trigger_tokens, $tokens );
	}

	/**
	 * validate
	 *
	 * @param mixed $trigger
	 * @param mixed $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		// 1. Basic checks - expecting booking, status, and event data
		if ( ! isset( $trigger['meta'][ $this->get_trigger_meta() ], $hook_args[0], $hook_args[1], $hook_args[2] ) ) {
			return false;
		}

		$booking    = $hook_args[0];
		$status     = $hook_args[1]['status'] ?? '';
		$event_data = $hook_args[2]; // Event data passed from our custom hook
		$event_id   = $event_data['id'] ?? null;

		// Set user ID to 0 to avoid issues with the user ID not being set.
		$this->set_user_id( 0 );

		// 2. Set user ID for Automator using the booking data
		$user_id = $this->get_user_id_from_booking( $booking );
		// If user ID is numeric, set it for Automator.
		if ( is_numeric( $user_id ) ) {
			$this->set_user_id( $user_id );
		}

		// 3. Compare event and status
		$selected_event  = $trigger['meta'][ $this->get_trigger_meta() ];
		$selected_status = $trigger['meta'][ $this->get_trigger_meta() . '_BOOKING_STATUS' ];

		// Early bail-out: if both are set to "Any", return true immediately
		if ( intval( '-1' ) === intval( $selected_event ) && intval( '-1' ) === intval( $selected_status ) ) {
			return true;
		}

		// Check if event matches (or "Any" is selected)
		if ( intval( '-1' ) !== intval( $selected_event ) && intval( $event_id ) !== intval( $selected_event ) ) {
			return false;
		}

		// Check if status matches (or "Any" is selected)
		if ( intval( '-1' ) !== intval( $selected_status ) && $status !== $selected_status ) {
			return false;
		}

		return true;
	}

	/**
	 * Get user ID from booking data
	 *
	 * @param array $booking
	 *
	 * @return int|false
	 */
	private function get_user_id_from_booking( $booking ) {
		// Try to get user ID from external ID first
		$external_id = $booking['customer']['externalId'] ?? null;
		if ( $external_id ) {
			$wp_user = get_user_by( 'ID', $external_id );
			if ( $wp_user ) {
				return $wp_user->ID;
			}
		}

		// Fallback to email lookup
		$email = isset( $booking['customer']['email'] ) ? $booking['customer']['email'] : null;
		if ( $email ) {
			$wp_user = get_user_by( 'email', $email );
			if ( $wp_user ) {
				return $wp_user->ID;
			}
		}

		return false;
	}

	/**
	 * hydrate_tokens
	 *
	 * @param mixed $completed_trigger
	 * @param mixed $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $completed_trigger, $hook_args ) {
		$status     = $hook_args[1]['status'] ?? '';
		$event_data = $hook_args[2]; // Event data passed from our custom hook

		return array(
			'EVENT_ID'             => $event_data['id'],
			'EVENT_TITLE'          => $event_data['name'],
			'EVENT_BOOKING_STATUS' => $status,
		);
	}
}
