<?php
/**
 * Trigger: ACF Group Subfield Updated
 *
 * @since <version>
 */

namespace Uncanny_Automator_Pro;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class ACF_GROUP_SUBFIELD_UPDATED
 */
class ACF_GROUP_SUBFIELD_UPDATED extends Trigger {

	const INTEGRATION_CODE = 'ACF';
	const TRIGGER_CODE     = 'ACF_GROUP_SUBFIELD_UPDATED_CODE';
	const TRIGGER_META     = 'ACF_GROUP_SUBFIELD_UPDATED_META';

	/**
	 * Setup the trigger.
	 *
	 * @return void
	 */
	public function setup_trigger(): void {
		$this->set_readable_sentence(
			esc_html_x(
				'{{A sub field}} in {{a group field}} is updated',
				'Advanced Custom Fields',
				'uncanny-automator-pro'
			)
		);

		$this->set_sentence(
			sprintf(
				/* translators: %1$s: Sub field, %2$s: Group field */
				esc_html_x(
					'{{A sub field:%1$s}} in {{a group field:%2$s}} is updated',
					'Advanced Custom Fields',
					'uncanny-automator-pro'
				),
				self::TRIGGER_META,
				'ACF_GROUP_FIELD:' . self::TRIGGER_META
			)
		);

		$this->set_integration( self::INTEGRATION_CODE );
		$this->set_trigger_code( self::TRIGGER_CODE );
		$this->set_trigger_meta( self::TRIGGER_META );
		$this->set_is_pro( true );
		$this->set_support_link(
			Automator()->get_author_support_link(
				self::TRIGGER_CODE,
				'integration/advanced-custom-fields/'
			)
		);

		$this->add_action( 'update_post_meta', 10, 4 );
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger Trigger definition.
	 * @param array $tokens  Tokens.
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ): array {
		$tokens[] = array(
			'tokenId'   => 'ACF_FIELD_NAME',
			'tokenName' => esc_html_x( 'Field name', 'Advanced Custom Fields', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		return $tokens;
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger   Trigger definition.
	 * @param array $hook_args Hook arguments.
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ): array {
		list( $id, $post_id, $meta_key, $meta_value ) = $hook_args;

		return array(
			'ACF_FIELD_NAME' => $meta_key,
		);
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options(): array {
		return array(
			array(
				'option_code' => 'ACF_GROUP_FIELD',
				'options_show_id' => false,
				'label'       => esc_html_x(
					'Group field',
					'Advanced Custom Fields',
					'uncanny-automator-pro'
				),
				'input_type'  => 'select',
				'required'    => true,
				'options'     => array(),
				'ajax'        => array(
					'endpoint' => 'uo_automator_acf_fetch_group_fields',
					'event'    => 'on_load',
				),
				'relevant_tokens' => array(),
			),
			array(
				'option_code' => self::TRIGGER_META,
				'options_show_id' => false,
				'label'       => esc_html_x(
					'Sub field',
					'Advanced Custom Fields',
					'uncanny-automator-pro'
				),
				'input_type'  => 'select',
				'required'    => true,
				'options'     => array(),
				'ajax'        => array(
					'endpoint'      => 'uo_automator_acf_fetch_sub_fields',
					'event'         => 'parent_fields_change',
					'listen_fields' => array( 'ACF_GROUP_FIELD' ),
				),
				'relevant_tokens' => array(),
			),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger Trigger definition.
	 * @param array $hook_args Hook arguments.
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ): bool {

		// Reject disallowed post types early.
		if ( in_array( $hook_args[2], Acf_Helpers_Pro::get_disallowed_post_types(), true ) ) {
			return false;
		}

		list( $id, $post_id, $meta_key, $meta_value ) = $hook_args;

		// Ignore WP internal meta.
		if ( '_edit_lock' === $meta_key || '_edit_last' === $meta_key ) {
			return false;
		}

		$selected_group_field = $trigger['meta']['ACF_GROUP_FIELD'] ?? '';
		$selected_sub_field   = $trigger['meta'][ self::TRIGGER_META ] ?? '';

		if ( '' === $selected_group_field || '' === $selected_sub_field ) {
			return false;
		}

		// Case 1: Any group + any subfield.
		if ( -1 === (int) $selected_group_field && -1 === (int) $selected_sub_field ) {
			static $pattern = null; // Cache the pattern for better performance.
			if ( null === $pattern ) {
				$needles = Acf_Helpers_Pro::get_group_field_names();
				if ( $needles ) {
					$pattern = '/' . implode( '|', array_map( 'preg_quote', $needles ) ) . '/i';
				}
			}
			return $pattern && preg_match( $pattern, $meta_key );
		}

		// Case 2: Specific group, any subfield.
		if ( -1 === (int) $selected_sub_field ) {
			return str_starts_with( $meta_key, $selected_group_field . '_' );
		}

		// Case 3: Specific group + exact subfield.
		$key_from_postmeta = $selected_group_field . '_' . $selected_sub_field;
		return $meta_key === $key_from_postmeta;
	}
}
