<?php
/**
 * Trigger: ACF Group Subfield Updated to Value
 *
 * @since <version>
 */

namespace Uncanny_Automator_Pro;

use Uncanny_Automator\Recipe\Trigger;

/**
 * Class ACF_GROUP_SUBFIELD_UPDATED_TO_VALUE
 */
class ACF_GROUP_SUBFIELD_UPDATED_TO_VALUE extends Trigger {

	const INTEGRATION_CODE = 'ACF';
	const TRIGGER_CODE     = 'ACF_GROUP_SUBFIELD_UPDATED_TO_VALUE_CODE';
	const TRIGGER_META     = 'ACF_GROUP_SUBFIELD_UPDATED_TO_VALUE_META';

	/**
	 * Setup the trigger.
	 *
	 * @return void
	 */
	public function setup_trigger(): void {
		$this->set_readable_sentence(
			esc_html_x(
				'{{A sub field}} in {{a group field}} is updated to {{a specific value}}',
				'Advanced Custom Fields',
				'uncanny-automator-pro'
			)
		);

		$this->set_sentence(
			sprintf(
				/* translators: %1$s: Sub field, %2$s: Group field, %3$s: Value */
				esc_html_x(
					'{{A sub field:%1$s}} in {{a group field:%2$s}} is updated to {{a specific value:%3$s}}',
					'Advanced Custom Fields',
					'uncanny-automator-pro'
				),
				self::TRIGGER_META,
				'ACF_GROUP_FIELD:' . self::TRIGGER_META,
				'ACF_VALUE:' . self::TRIGGER_META
			)
		);

		$this->set_integration( self::INTEGRATION_CODE );
		$this->set_trigger_code( self::TRIGGER_CODE );
		$this->set_trigger_meta( self::TRIGGER_META );
		$this->set_is_pro( true );
		$this->set_support_link(
			Automator()->get_author_support_link(
				self::TRIGGER_CODE,
				'integration/advanced-custom-fields/'
			)
		);

		$this->add_action( 'update_post_meta', 10, 4 );
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger Trigger definition.
	 * @param array $tokens  Tokens.
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ): array {

		$tokens[] = array(
			'tokenId'   => 'ACF_FIELD_NAME',
			'tokenName' => esc_html_x( 'Field name', 'Advanced Custom Fields', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		$tokens[] = array(
			'tokenId'   => 'ACF_MATCHED_VALUE',
			'tokenName' => esc_html_x( 'Matched value', 'Advanced Custom Fields', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		return $tokens;
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $trigger   Trigger definition.
	 * @param array $hook_args Hook arguments.
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ): array {

		list( $id, $post_id, $meta_key, $meta_value ) = $hook_args;

		$hydrated = array(
			'ACF_FIELD_NAME'    => $meta_key,
			'ACF_MATCHED_VALUE' => $meta_value,
		);

		return $hydrated;
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options(): array {
		$group_field = array(
			'option_code'     => 'ACF_GROUP_FIELD',
			'options_show_id' => false,
			'label'           => esc_html_x(
				'Group field',
				'Advanced Custom Fields',
				'uncanny-automator-pro'
			),
			'input_type'      => 'select',
			'required'        => true,
			'options'         => array(),
			'ajax'            => array(
				'endpoint' => 'uo_automator_acf_fetch_group_fields',
				'event'    => 'on_load',
			),
			'relevant_tokens' => array(),
		);

		$sub_field = array(
			'option_code'     => self::TRIGGER_META,
			'options_show_id' => false,
			'label'           => esc_html_x(
				'Sub field',
				'Advanced Custom Fields',
				'uncanny-automator-pro'
			),
			'input_type'      => 'select',
			'required'        => true,
			'options'         => array(),
			'ajax'            => array(
				'endpoint'      => 'uo_automator_acf_fetch_sub_fields',
				'event'         => 'parent_fields_change',
				'listen_fields' => array( 'ACF_GROUP_FIELD' ),
			),
			'relevant_tokens' => array(),
		);

		$value_field = array(
			'option_code'     => 'ACF_VALUE',
			'label'           => esc_html_x(
				'Value to match',
				'Advanced Custom Fields',
				'uncanny-automator-pro'
			),
			'input_type'      => 'text',
			'required'        => true,
			'relevant_tokens' => array(),
		);

		return array( $group_field, $sub_field, $value_field );
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger Trigger definition.
	 * @param array $hook_args Hook arguments.
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ): bool {

		// Reject disallowed post types early.
		if ( in_array( $hook_args[2], Acf_Helpers_Pro::get_disallowed_post_types(), true ) ) {
			return false;
		}

		list( $id, $post_id, $meta_key, $meta_value ) = $hook_args;

		// Ignore WP internal meta.
		if ( '_edit_lock' === $meta_key || '_edit_last' === $meta_key ) {
			return false;
		}

		$selected_group_field = $trigger['meta']['ACF_GROUP_FIELD'] ?? '';
		$selected_sub_field   = $trigger['meta'][ self::TRIGGER_META ] ?? '';
		$selected_value       = $trigger['meta']['ACF_VALUE'] ?? '';

		if ( '' === $selected_group_field || '' === $selected_sub_field || '' === $selected_value ) {
			return false;
		}

		// Case 1: Any group + any subfield + value match.
		if ( -1 === (int) $selected_group_field && -1 === (int) $selected_sub_field ) {
			static $pattern = null; // Cache the pattern for better performance.
			if ( null === $pattern ) {
				$needles = Acf_Helpers_Pro::get_group_field_names();
				if ( $needles ) {
					$pattern = '/' . implode( '|', array_map( 'preg_quote', $needles ) ) . '/i';
				}
			}
			return $pattern && preg_match( $pattern, $meta_key )
				&& $this->value_matches( $meta_value, $selected_value );
		}

		// Case 2: Specific group, any subfield + value match.
		if ( -1 === (int) $selected_sub_field ) {
			return str_starts_with( $meta_key, $selected_group_field . '_' )
				&& $this->value_matches( $meta_value, $selected_value );
		}

		// Case 3: Specific group + exact subfield + value match.
		$key_from_postmeta = $selected_group_field . '_' . $selected_sub_field;
		return $meta_key === $key_from_postmeta
			&& $this->value_matches( $meta_value, $selected_value );
	}

	/**
	 * Check if meta value matches selected value.
	 *
	 * @param mixed  $meta_value     The actual field value.
	 * @param string $selected_value The value to match against.
	 * @return bool
	 */
	private function value_matches( $meta_value, $selected_value ): bool {

		// Handle array values (multi-select, checkboxes, etc.).
		if ( is_array( $meta_value ) ) {
			// Check if selected value is in array
			if ( in_array( $selected_value, $meta_value, true ) ) {
				return true;
			}
			// Also check comma-separated string
			$meta_value = implode( ', ', $meta_value );
		}

		// Trim whitespace and convert to strings for comparison.
		$meta_value_str     = trim( (string) $meta_value );
		$selected_value_str = trim( (string) $selected_value );

		return $meta_value_str === $selected_value_str;
	}
}
