<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName
namespace Uncanny_Automator_Pro;

/**
 * Class Acf_Helpers
 *
 * @package Uncanny_Automator
 */
class Acf_Helpers_Pro {

	/**
	 * Options.
	 *
	 * @var mixed $options The options.
	 */
	public $options;

	/**
	 * Pro.
	 *
	 * @var mixed $pro The pro helper options.
	 */
	public $pro;

	/**
	 * Load options.
	 *
	 * @var bool
	 */
	public $load_options = true;

	public static $instance;

	/**
	 * Class constructor.
	 *
	 * @return void
	 */
	public function __construct() {

		$this->load_options = Automator()->helpers->recipe->maybe_load_trigger_options( __CLASS__ );

		add_action( 'wp_ajax_uo_automator_acf_get_fields', array( $this, 'acf_get_fields' ) );
		add_action( 'wp_ajax_uo_automator_acf_get_post_under_post_type', array( $this, 'acf_get_post_under_post_type' ) );
		add_action( 'wp_ajax_uo_automator_acf_fetch_sub_fields', array( $this, 'fetch_sub_fields' ) );
		add_action( 'wp_ajax_uo_automator_acf_fetch_group_fields', array( $this, 'fetch_group_fields' ) );
	}

	/**
	 *
	 * Get instance.
	 *
	 * @return mixed
	 */
	public static function get_instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Get disallowed post types.
	 *
	 * @return mixed|void
	 */
	public static function get_disallowed_post_types() {

		$disallowed_post_types = array(
			'attachment',
			'revision',
			'nav_menu_item',
			'custom_css',
			'customize_changeset',
			'oembed_cache',
			'user_request',
			'wp_block',
			'wp_template',
		);

		return apply_filters( 'uo_automator_acf_disallowed_post_types', $disallowed_post_types );
	}

	/**
	 * Set options.
	 *
	 * @param Acf_Helpers_Pro $options
	 */
	public function setOptions( Acf_Helpers_Pro $options ) { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.MethodNameInvalid
		$this->options = $options;
	}

	/**
	 * Set pro.
	 *
	 * @param Acf_Pro_Helpers $pro
	 */
	public function setPro( Acf_Helpers_Pro $pro ) { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.MethodNameInvalid
		$this->pro = $pro;
	}

	/**
	 * Outputs all the post under the selected post type.
	 *
	 * @return void
	 */
	public function acf_get_post_under_post_type() {

		$list = array();

		$selected_post_type = automator_filter_input( 'value', INPUT_POST );
		$recipe_id          = automator_filter_input( 'recipe_id', INPUT_POST );

		$args = array(
			'post_type'      => $selected_post_type,
			'posts_per_page' => apply_filters( 'automator_acf_post_type_posts_per_page', 9999, $selected_post_type, $recipe_id ),
			'orderby'        => 'title',
			'order'          => 'ASC',
		);

		$obj = get_post_type_object( $selected_post_type );

		$post_type_name = esc_html_x( 'post', 'Advanced Custom Fields', 'uncanny-automator' );

		if ( isset( $obj->labels->singular_name ) ) {
			$post_type_name = $obj->labels->singular_name;
		}

		$query = new \WP_Query( $args );

		$list[] = array(
			'value' => $selected_post_type,
			/* translators: The post type singular label */
			'text'  => sprintf( esc_html_x( 'Any %s', 'Advanced Custom Fields', 'uncanny-automator' ), strtolower( $post_type_name ) ),
		);

		if ( $query->have_posts() ) :

			while ( $query->have_posts() ) :
				$query->the_post();

				$list[] = array(
					'value' => get_the_ID(),
					'text'  => get_the_title(),
				);

			endwhile;

			wp_reset_postdata();

		endif;

		wp_send_json( $list );
	}

	/**
	 * Outputs all the field under a selected post.
	 *
	 * @return void.
	 */
	public function acf_get_fields() {

		$post_id = automator_filter_input( 'value', INPUT_POST );

		$args = array(
			'post_id' => $post_id,
		);

		// Treat it as post type if its non-numeric.
		if ( ! is_numeric( $post_id ) ) {
			$args = array(
				'post_type' => $post_id,
			);
		}

		$field_groups_collection = acf_get_field_groups( $args );

		$field_groups = array();

		foreach ( $field_groups_collection as $field_group ) {

			$field_groups[] = acf_get_fields( $field_group['key'] );

		}

		$singular_fields[] = array(
			'value' => '-1',
			'text'  => esc_html_x( 'Any field', 'Advanced Custom Fields', 'uncanny-automator' ),
		);

		if ( ! empty( $field_groups ) && is_array( $field_groups ) ) {

			foreach ( $field_groups as $field_groups ) {

				foreach ( $field_groups as $field_group ) {

					$singular_fields[] = array(
						'value' => $field_group['name'],
						'text'  => $field_group['label'],
					);

				}
			}
		}

		wp_send_json( $singular_fields );
	}

	/**
	 * Get the ACF field on 'user_form' location.
	 **/

	public function get_user_fields() {

		if ( ! function_exists( 'acf_get_fields' ) ) {
			return array();
		}

		$fields = array(
			'-1' => esc_html_x( 'Any field', 'Advanced Custom Fields', 'uncanny-automator' ),
		);

		$field_groups = $this->get_user_form_field_groups();

		if ( empty( $field_groups ) ) {

			return array();

		}

		foreach ( $field_groups as $group ) {

			$group_fields = acf_get_fields( $group['key'] );

			if ( ! empty( $group_fields ) ) {

				foreach ( $group_fields as $field ) {

					$fields[ esc_attr( $field['name'] ) ] = esc_html( $field['label'] );

				}
			}
		}

		return $fields;
	}

	/**
	 * Get all field groups where location is equals to 'user_form'.
	 *
	 * @return array The field groups.
	 */
	public function get_user_form_field_groups() {

		if ( ! function_exists( 'acf_get_field_groups' ) ) {

			return array();

		}

		$groups_user_form = array();

		$field_groups = acf_get_field_groups();

		foreach ( $field_groups as $group ) {

			if ( ! empty( $group['location'] ) ) {

				foreach ( $group['location'] as $locations ) {

					foreach ( $locations as $location ) {

						if ( 'user_role' === $location['param'] || 'user_form' === $location['param'] ) {

							$groups_user_form[] = $group;

						}
					}
				}
			}
		}

		return $groups_user_form;
	}

	/**
	 * Get acf group fields.
	 *
	 * @return mixed
	 */
	public function fetch_group_fields() {

		Automator()->utilities->verify_nonce();

		if ( ! function_exists( 'acf_get_field_groups' ) ) {
			return array();
		}

		// Defaults to any group field.
		$options = array(
			array(
				'value' => '-1',
				'text'  => esc_html_x( 'Any group field', 'Advanced Custom Fields', 'uncanny-automator-pro' ),
			),
		);

		$field_groups = acf_get_field_groups();

		foreach ( $field_groups as $field_group ) {
			$fields = acf_get_fields( $field_group['key'] );
			if ( ! empty( $fields ) ) {
				foreach ( $fields as $field ) {
					if ( 'group' === $field['type'] && isset( $field['name'], $field['label'] ) ) {
						$options[] = array(
							'value' => $field['name'],
							'text'  => $field['label'],
						);
					}
				}
			}
		}

		wp_send_json(
			array(
				'success' => true,
				'options' => $options,
			)
		);
	}

	/**
	 * Fetch sub fields via AJAX.
	 *
	 * @return void
	 */
	public function fetch_sub_fields() {

		Automator()->utilities->verify_nonce();

		if ( ! function_exists( 'acf_get_field' ) ) {
			wp_send_json(
				array(
					'success' => false,
					'message' => esc_html_x( 'ACF is not active', 'Advanced Custom Fields', 'uncanny-automator-pro' ),
				)
			);
		}

		$request_vars = automator_filter_input_array( 'values', INPUT_POST );
		$group_key    = $request_vars['ACF_GROUP_FIELD'] ?? '';

		$options = array(
			array(
				'text'  => esc_html_x( 'Any sub field', 'Advanced Custom Fields', 'uncanny-automator-pro' ),
				'value' => '-1',
			),
		);

		if ( -1 === intval( $group_key ) || '' === $group_key ) {
			wp_send_json(
				array(
					'success' => true,
					'options' => $options,
				)
			);
		}

		$group_field = acf_get_field( $group_key );
		if ( $group_field && isset( $group_field['sub_fields'] ) ) {
			foreach ( (array) $group_field['sub_fields'] as $sub_field ) {
				if ( ! isset( $sub_field['label'], $sub_field['name'] ) ) {
					continue;
				}
				$options[] = array(
					'value' => $sub_field['name'],
					'text'  => $sub_field['label'],
				);
			}
		}

		wp_send_json(
			array(
				'success' => true,
				'options' => $options,
			)
		);
	}

	/**
	 * Get flat array of ACF group field names.
	 *
	 * @return array
	 */
	public static function get_group_field_names() {

		if ( ! function_exists( 'acf_get_field_groups' ) ) {
			return array();
		}

		$group_names  = array();
		$field_groups = acf_get_field_groups();

		foreach ( $field_groups as $field_group ) {
			$fields = acf_get_fields( $field_group['key'] );
			if ( ! empty( $fields ) ) {
				foreach ( $fields as $field ) {
					if ( 'group' === $field['type'] && isset( $field['name'] ) ) {
						$group_names[] = $field['name'];
					}
				}
			}
		}

		return $group_names;
	}
}
