<?php
namespace Uncanny_Automator_Pro\Loops\Loop\Model\Query;

use wpdb;

/**
 * Handles reads/writes to the uap_loop_entries_items_data table.
 *
 * @since 6.3.0
 */
class Loop_Entry_Item_Data_Query {

	/**
	 * @var wpdb $db
	 */
	private $db = null;

	/**
	 * @var string $table
	 */
	private $table = '';

	/**
	 * Sets the db and table name.
	 */
	public function __construct() {
		global $wpdb;
		$this->db    = $wpdb;
		$this->table = $this->db->prefix . 'uap_loop_entries_items_data';
	}

	/**
	 * Finds a data row by action id and meta hash.
	 *
	 * @param int $action_id The action post ID.
	 * @param string $meta_hash The sha256 hash of the serialized payload.
	 *
	 * @return int|null The row ID or null if not found.
	 */
	public function find_by_action_and_hash( $action_id, $meta_hash ) {
		$id = $this->db->get_var(
			$this->db->prepare(
				"SELECT ID FROM {$this->table} WHERE action_id = %d AND meta_hash = %s",
				absint( $action_id ),
				(string) $meta_hash
			)
		);

		return $id ? absint( $id ) : null;
	}

	/**
	 * Inserts the payload if it does not exist. Returns the ID of the row.
	 *
	 * Payload structure stores only the essentials for token parsing.
	 *
	 * @param int   $action_id The action post ID.
	 * @param array $meta_array Flattened meta key=>value pairs.
	 *
	 * @return int The row ID (existing or newly inserted). Returns 0 on failure.
	 */
	public function find_or_create_by_action_meta( $action_id, $meta_array ) {
		$payload    = array( 'action_data' => array( 'meta' => (array) $meta_array ) );
		$serialized = maybe_serialize( $payload );
		$meta_hash  = hash( 'sha256', (string) $serialized );

		// Fast-path: return existing ID when present.
		$existing_id = $this->find_by_action_and_hash( $action_id, $meta_hash );
		if ( $existing_id ) {
			return $existing_id;
		}

		// Robust upsert to avoid race conditions.
		// Use ON DUPLICATE KEY UPDATE to set LAST_INSERT_ID to existing row without modifying data.
		$query = $this->db->prepare(
			"INSERT INTO {$this->table} (action_id, meta_hash, data, date_added)
             VALUES (%d, %s, %s, %s)
             ON DUPLICATE KEY UPDATE ID = LAST_INSERT_ID(ID)",
			absint( $action_id ),
			(string) $meta_hash,
			(string) $serialized,
			current_time( 'mysql' )
		);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$result = $this->db->query( $query );

		if ( false === $result ) {
			// Fallback: another process likely inserted the row; fetch its ID.
			$fallback_id = $this->find_by_action_and_hash( $action_id, $meta_hash );
			return $fallback_id ? $fallback_id : 0;
		}

		return absint( $this->db->insert_id );
	}

	/**
	 * Finds a row by its primary ID.
	 *
	 * @param int $id The primary key.
	 *
	 * @return array|null Associative array with columns or null if not found.
	 */
	public function find_by_id( $id ) {
		$row = $this->db->get_row(
			$this->db->prepare( "SELECT * FROM {$this->table} WHERE ID = %d", absint( $id ) ),
			ARRAY_A
		);

		return is_array( $row ) ? $row : null;
	}

	/**
	 * Finds a row by ID.
	 *
	 * @param int $id The ID.
	 *
	 * @return array|null Associative array with columns or null if not found.
	 */
	public function find_action_data_by_id( $id ) {
		$row = $this->db->get_var(
			$this->db->prepare( "SELECT data as `action_data` FROM {$this->table} WHERE id = %d", absint( $id ) )
		);

		return $row ? maybe_unserialize( $row ) : null;
	}
}
