<?php

namespace Uncanny_Automator_Pro\Settings;

use Uncanny_Automator_Pro\Settings\Pro_Premium_Integration_Items;
use Uncanny_Automator\Settings\Settings_Options;
use Uncanny_Automator_Pro\Settings\Pro_Integration_Templating;
use Uncanny_Automator\Settings\Premium_Integration_Rest_Processing;
use Uncanny_Automator\Settings\App_Integration_Settings_Setup;
use Uncanny_Automator\Settings\Premium_Integration_Alerts;

/**
 * Abstract class for Pro settings that extend free integration settings
 *
 * This class provides a standardized way to extend free integration settings
 * with Pro functionality while maintaining consistency across integrations.
 *
 * @package Uncanny_Automator_Pro\Core\Classes
 */
abstract class App_Integration_Pro_Settings {

	use Pro_Integration_Templating;
	use Pro_Premium_Integration_Items;
	use Settings_Options;
	use Premium_Integration_Rest_Processing;
	use App_Integration_Settings_Setup;
	use Premium_Integration_Alerts;

	/**
	 * App Helpers instance for this integration.
	 *
	 * @var App_Helpers
	 */
	protected $helpers;

	/**
	 * API instance for this integration.
	 *
	 * @var Api_Caller
	 */
	protected $api;

	/**
	 * Webhooks instance for this integration.
	 *
	 * @var App_Webhooks|null
	 */
	protected $webhooks;

	/**
	 * Integration dependencies.
	 *
	 * @var object
	 * @property App_Helpers helpers
	 * @property Api_Caller api
	 * @property App_Webhooks webhooks|null
	 */
	protected $dependencies;

	/**
	 * Settings configuration for this integration.
	 *
	 * @var array
	 */
	protected $settings = array();

	/**
	 * @var bool Whether to append "(when enabled)" to trigger names.
	 */
	protected $append_trigger_enabled_suffix = false;

	/**
	 * @var string The settings ID.
	 */
	protected $settings_id;

	/**
	 * Whether the integration is a third party integration
	 *
	 * @var null\boolean - null if not set
	 */
	public $is_third_party = null;

	/**
	 * Constructor.
	 *
	 * @param  object $dependencies Dependencies object
	 * @param  array $settings     Optional settings array
	 *
	 * @return void
	 */
	final public function __construct( $dependencies, $settings = array() ) {

		if ( ! empty( $dependencies ) ) {
			$this->dependencies = $dependencies;

			// Set direct properties for clean access.
			if ( is_object( $this->dependencies ) ) {
				$this->helpers  = $this->dependencies->helpers ?? null;
				$this->api      = $this->dependencies->api ?? null;
				$this->webhooks = $this->dependencies->webhooks ?? null;
			}
		}

		// Store settings for later use
		$this->settings = $settings;

		// Setup the settings page properties.
		$this->setup_settings_properties();

		// Register the hooks.
		$this->register_settings_hooks();
	}

	/**
	 * Setup the settings page properties from the config.
	 *
	 * @return void
	 * @throws \Exception
	 */
	public function setup_settings_properties() {

		// Set settings if provided
		if ( ! empty( $this->settings ) ) {
			$this->set_settings( $this->settings );
			$this->settings_id = $this->get_id();
		}

		// Allow extending class to set additional properties.
		$this->set_properties();

		// Connection status specific properties.
		if ( $this->is_connected ) {
			$this->set_connected_properties();
		} else {
			$this->set_disconnected_properties();
		}
	}

	/**
	 * Register base hooks that are common to all Pro settings
	 *
	 * @return void
	 */
	protected function register_settings_hooks() {

		// Hook into the Premium_Integration_Items trait's filter for triggers.
		add_filter(
			"automator_{$this->settings_id}_triggers",
			array( $this, 'add_pro_triggers' ),
			10,
			1
		);

		// Hook into the Premium_Integration_Items trait's filter for actions.
		add_filter(
			"automator_{$this->settings_id}_actions",
			array( $this, 'add_pro_actions' ),
			10,
			1
		);

		// Register this Pro settings instance with Action_Manager for REST handling.
		// We register with the original ID but with a Pro suffix to map to the correct settings object.
		add_filter(
			'automator_integration_settings_instance_' . sanitize_key( $this->get_pro_id() ),
			function () {
				return $this;
			}
		);

		// Hook into before disconnect
		add_filter(
			'automator_before_disconnect_' . $this->get_id(),
			array( $this, 'before_disconnect_pro_cleanup' ),
			10,
			3
		);

		// Hook into after disconnect
		add_filter(
			'automator_after_disconnect_' . $this->get_id(),
			array( $this, 'after_disconnect_pro_cleanup' ),
			10,
			3
		);

		// Register Pro templating hooks
		$this->register_pro_templating_hooks();

		// Enqueue the assets
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );

		// Register the options for dynamic sanitization & saving.
		$this->option_registration();

		// Register the hooks.
		$this->register_hooks();
	}

	/**
	 * Add Pro triggers to the list
	 *
	 * @param array $triggers The existing triggers
	 * @return array
	 */
	public function add_pro_triggers( $triggers ) {
		// Get Pro triggers using the trait's functionality
		$pro_triggers = $this->get_items_from_classes( 'triggers' );

		// Append "(when enabled)" to each trigger if enabled
		if ( $this->append_trigger_enabled_suffix ) {
			$pro_triggers = array_map(
				function ( $trigger ) {
					return $trigger . ' ' . esc_html_x( '( when enabled )', 'Integration settings', 'uncanny-automator-pro' );
				},
				$pro_triggers
			);
		}

		return array_merge( $triggers, $pro_triggers );
	}

	/**
	 * Add Pro actions to the list
	 *
	 * @param array $actions The existing actions
	 * @return array
	 */
	public function add_pro_actions( $actions ) {
		// Get Pro actions using the trait's functionality
		$pro_actions = $this->get_items_from_classes( 'actions' );

		return array_merge( $actions, $pro_actions );
	}

	/**
	 * Output Pro save button with correct integration ID
	 *
	 * @param string $action The action name
	 * @param string $label The button label
	 * @param array $attributes Additional attributes
	 * @return void
	 */
	protected function output_pro_action_button( $action, $label, $attributes = array() ) {
		$default_attributes = array(
			'integration-id' => $this->get_pro_id(),
		);

		$attributes = array_merge( $default_attributes, $attributes );

		$this->output_action_button( $action, $label, $attributes );
	}

	/**
	 * Output Pro save button with default settings
	 *
	 * @param string $action The action name (defaults to 'pro_save_settings')
	 * @param string $label The button label (defaults to 'Save settings')
	 * @param array $attributes Additional attributes
	 * @return void
	 */
	public function output_pro_save_button( $action = 'pro_save_settings', $label = null, $attributes = array() ) {
		if ( null === $label ) {
			$label = esc_html_x(
				'Save settings',
				'Integration settings',
				'uncanny-automator-pro'
			);
		}

		$this->output_pro_action_button( $action, $label, $attributes );
	}

	/**
	 * Set whether to append "(when enabled)" suffix to trigger names
	 *
	 * @param bool $append Whether to append the suffix
	 * @return void
	 */
	protected function set_append_trigger_enabled_suffix( $append ) {
		$this->append_trigger_enabled_suffix = (bool) $append;
	}

	/**
	 * Get the Pro settings ID with '-pro' suffix
	 *
	 * @return string
	 */
	protected function get_pro_id() {
		return $this->get_id() . '-pro';
	}

	/**
	 * Perform cleanup before disconnection
	 * Override this method in extending classes to add custom cleanup
	 *
	 * @param array $response The current response array
	 * @param array $data The posted data
	 * @param object $base_settings_object The base settings object
	 *
	 * @return array Modified response array
	 */
	public function before_disconnect_pro_cleanup( $response, $data, $base_settings_object ) {
		// Default implementation returns response unchanged
		// Extending classes can override this method to add custom cleanup
		return $response;
	}

	/**
	 * Perform cleanup after disconnection
	 * Override this method in extending classes to add custom cleanup
	 *
	 * @param array $response The current response array
	 * @param array $data The posted data
	 * @param object $base_settings_object The base settings object
	 *
	 * @return array Modified response array
	 */
	public function after_disconnect_pro_cleanup( $response, $data, $base_settings_object ) {
		// Default implementation returns response unchanged
		// Extending classes can override this method to add custom cleanup
		return $response;
	}
}
